﻿using System;
using System.Collections.Generic;
using NUnit.Framework;

namespace Magify.Tests
{
    public class AbstractProductsCollectionTests
    {
        #region One Product

        [Test]
        [UnitTestTarget(typeof(AbstractProductsCollection), nameof(AbstractProductsCollection.PullStateFrom))]
        public void OneProductWasReplaced()
        {
            // Arrange
            var products = new List<InAppProduct>
            {
                new("in_app_product_0", false, PurchaseStore.Native),
                new("in_app_product_1", true, PurchaseStore.Native),
                new("in_app_product_2", false, PurchaseStore.Native),
            };
            var abstractProducts = new AbstractProductsCollection(products);
            var updatedProducts = new List<InAppProduct>
            {
                new("in_app_product_0", false, PurchaseStore.Native),
                new("in_app_product_1", true, PurchaseStore.Native),
                new("in_app_product_ultra_0", false, PurchaseStore.Native),
            };

            // Act
            abstractProducts.PullStateFrom(updatedProducts);

            // Assert
            Assert.IsTrue(IsProductsListsSame(products, updatedProducts) && products.Count == 3);
        }

        [Test]
        [UnitTestTarget(typeof(AbstractProductsCollection), nameof(AbstractProductsCollection.PullStateFrom))]
        public void OneProductWasDeleted()
        {
            // Arrange
            var products = new List<InAppProduct>
            {
                new("in_app_product_0", false, PurchaseStore.Native),
                new("in_app_product_1", true, PurchaseStore.Native),
                new("in_app_product_2", false, PurchaseStore.Native),
            };
            var abstractProducts = new AbstractProductsCollection(products);
            var updatedProducts = new List<InAppProduct>
            {
                new("in_app_product_0", false, PurchaseStore.Native),
                new("in_app_product_1", true, PurchaseStore.Native),
            };

            // Act
            abstractProducts.PullStateFrom(updatedProducts);

            // Assert
            Assert.IsTrue(IsProductsListsSame(products, updatedProducts) && products.Count == 2);
        }

        [Test]
        [UnitTestTarget(typeof(AbstractProductsCollection), nameof(AbstractProductsCollection.PullStateFrom))]
        public void OneProductWasAdded()
        {
            // Arrange
            var products = new List<InAppProduct>
            {
                new("in_app_product_0", false, PurchaseStore.Native),
                new("in_app_product_1", true, PurchaseStore.Native),
            };
            var abstractProducts = new AbstractProductsCollection(products);
            var updatedProducts = new List<InAppProduct>
            {
                new("in_app_product_0", false, PurchaseStore.Native),
                new("in_app_product_1", true, PurchaseStore.Native),
                new("in_app_product_2", false, PurchaseStore.Native),
            };

            // Act
            abstractProducts.PullStateFrom(updatedProducts);

            // Assert
            Assert.IsTrue(IsProductsListsSame(products, updatedProducts) && products.Count == 3);
        }

        #endregion

        #region Many products

        [Test]
        [UnitTestTarget(typeof(AbstractProductsCollection), nameof(AbstractProductsCollection.PullStateFrom))]
        public void ManyProductWasReplaced()
        {
            // Arrange
            var products = new List<InAppProduct>
            {
                new("in_app_product_0", false, PurchaseStore.Native),
                new("in_app_product_1", true, PurchaseStore.Native),
                new("in_app_product_2", false, PurchaseStore.Native),
            };
            var abstractProducts = new AbstractProductsCollection(products);
            var updatedProducts = new List<InAppProduct>
            {
                new("in_app_product_changed_5", true, PurchaseStore.Native),
                new("in_app_product_changed_8", false, PurchaseStore.Native),
                new("in_app_product_changed_15", true, PurchaseStore.Native),
            };

            // Act
            abstractProducts.PullStateFrom(updatedProducts);

            // Assert
            Assert.IsTrue(IsProductsListsSame(products, updatedProducts) && products.Count == 3);
        }

        [Test]
        [UnitTestTarget(typeof(AbstractProductsCollection), nameof(AbstractProductsCollection.PullStateFrom))]
        public void ManyProductWasDeleted()
        {
            // Arrange
            var products = new List<InAppProduct>
            {
                new("in_app_product_0", false, PurchaseStore.Native),
                new("in_app_product_1", true, PurchaseStore.Native),
                new("in_app_product_2", false, PurchaseStore.Native),
                new("in_app_product_changed_5", true, PurchaseStore.Native),
                new("in_app_product_changed_8", false, PurchaseStore.Native),
                new("in_app_product_changed_15", true, PurchaseStore.Native),
            };
            var abstractProducts = new AbstractProductsCollection(products);
            var updatedProducts = new List<InAppProduct>
            {
                new("in_app_product_0", false, PurchaseStore.Native),
            };

            // Act
            abstractProducts.PullStateFrom(updatedProducts);

            // Assert
            Assert.IsTrue(IsProductsListsSame(products, updatedProducts) && products.Count == 1);
        }

        [Test]
        [UnitTestTarget(typeof(AbstractProductsCollection), nameof(AbstractProductsCollection.PullStateFrom))]
        public void ManyProductWasAdded()
        {
            // Arrange
            var products = new List<InAppProduct>
            {
                new("in_app_product_0", false, PurchaseStore.Native),
            };
            var abstractProducts = new AbstractProductsCollection(products);
            var updatedProducts = new List<InAppProduct>
            {
                new("in_app_product_0", false, PurchaseStore.Native),
                new("in_app_product_1", true, PurchaseStore.Native),
                new("in_app_product_2", false, PurchaseStore.Native),
                new("in_app_product_changed_5", true, PurchaseStore.Native),
                new("in_app_product_changed_8", false, PurchaseStore.Native),
                new("in_app_product_changed_15", true, PurchaseStore.Native),
            };

            // Act
            abstractProducts.PullStateFrom(updatedProducts);

            // Assert
            Assert.IsTrue(IsProductsListsSame(products, updatedProducts) && products.Count == 6);
        }

        #endregion

        private static bool IsProductsListsSame<T>(List<T> products, List<T> updatedProducts) where T : ProductDef
        {
            var lengthSame = products.Count == updatedProducts.Count;
            if (!lengthSame)
            {
                return false;
            }

            Comparison<T> comparison = (productA, productB) => string.CompareOrdinal(productA.Id, productB.Id);
            products.Sort(comparison);
            updatedProducts.Sort(comparison);
            for (var i = 0; i < products.Count; i++)
            {
                if (products[i].ToJson() != updatedProducts[i].ToJson())
                {
                    return false;
                }
            }
            return true;
        }
    }
}