using System.Threading;
using System.Threading.Tasks;
using FluentAssertions;
using JetBrains.Annotations;
using NSubstitute;
using NUnit.Framework;

namespace Magify.Tests
{
    public class AppStateManagerTests
    {
        [NotNull]
        private static object[] HasSocialAuthorizeCases => new object[]
        {
            new object[] { "", "", false },
            new object[] { "", "token", false },
            new object[] { "provider", "", false },
            new object[] { "provider", "token", true },
        };

        [Test]
        [TestCaseSource(nameof(HasSocialAuthorizeCases))]
        public async Task WhenSocialAuthorize_ThenShouldBeAuthorized([NotNull] string provider, [NotNull] string token, bool expectedState)
        {
            //Arrange
            using var _ = new TemporaryIgnoreFailingLogMessagesScope();
            using var subsystems = CreateAppStateManager(out var appStateManager);
            subsystems.ForEach<IInitializable>(s => s!.Initialize());
            var cts = new CancellationTokenSource(200);

            //Act
            appStateManager.SyncStateEnabled.Value = true;
            await appStateManager.RequestSocialAuthTokenFor(provider, token, cts.Token);

            //Assert
            appStateManager.HasSocialAuthorizationData.Should()!.Be(expectedState);
        }

        [NotNull]
        private static SubsystemsCollection CreateAppStateManager([NotNull] out AppStateManager appStateManager)
        {
            var subsystems = new SubsystemsCollection();
            var platformApi = new PlatformDefault().AddTo(subsystems);
            var serverApi = Substitute.For<IServerApi>()!;
            var appVersionProvider = new AppVersionProvider().AddTo(subsystems);
            var generalPrefs = GeneralPrefs.Create(EditorModeTestsEnvironment.GeneralPrefsPath, EditorModeTestsEnvironment.LocalGeneralPrefsPath, appVersionProvider).AddTo(subsystems);
            var appStatePrefs = AppStatePrefs.Create(EditorModeTestsEnvironment.AppStatePrefsPath).AddTo(subsystems);
            var clientIdProvider = new ClientIdProvider(generalPrefs, appStatePrefs, platformApi, customClientId: null).AddTo(subsystems);
            var counters = new CountersStorage(EditorModeTestsEnvironment.CountersFolderPath).AddTo(subsystems);
            var clientStateConfig = EditorModeTestsEnvironment.DefaultClientStateConfig.AddTo(subsystems);
            var storagePath = EditorModeTestsEnvironment.TempStoragePath;
            appStateManager = new AppStateManager(platformApi, serverApi, generalPrefs, clientIdProvider, counters, appStatePrefs, clientStateConfig, storagePath);
            return subsystems;
        }
    }
}