using System;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;

namespace Magify
{
    /// <summary>
    /// Not inherently thread-safe. Synchronization is required by the caller when accessing the reference concurrently.
    /// </summary>
    internal static class ThreadingUtils
    {
        public static void CancelAndReplace(ref CancellationTokenSource cts)
        {
            var old = cts;
            cts = new CancellationTokenSource();
            old?.Cancel();
            old?.Dispose();
        }

        public static void CancelAndReplace<T>(ref CancellationTokenSource cts, T newValue)
            where T: CancellationTokenSource
        {
            var old = cts;
            cts = newValue;
            old?.Cancel();
            old?.Dispose();
        }

        public static void CancelAndRemove(ref CancellationTokenSource cts)
        {
            var old = cts;
            cts = null;
            old?.Cancel();
            old?.Dispose();
        }

        public static void CancelAndRemove(ref CancellationTokenSource cts1, ref CancellationTokenSource cts2)
        {
            var old1 = cts1;
            var old2 = cts2;

            cts1 = null;
            cts2 = null;

            old1?.Cancel();
            old2?.Cancel();
            old1?.Dispose();
            old2?.Dispose();
        }

        public static void CancelAndReplace(ref CancellationTokenSource cts1, ref CancellationTokenSource cts2)
        {
            var old1 = cts1;
            var old2 = cts2;

            cts1 = new CancellationTokenSource();
            cts2 = new CancellationTokenSource();

            old1?.Cancel();
            old2?.Cancel();
            old1?.Dispose();
            old2?.Dispose();
        }

        public static void TrySetResultAndRemove<T>(ref TaskCompletionSource<T> completionSource, [CanBeNull] T result)
        {
            var old = completionSource;
            completionSource = null;
            old?.TrySetResult(result);
        }

        public static void TryCancelAndRemove<T>(ref TaskCompletionSource<T> completionSource)
        {
            var old = completionSource;
            completionSource = null;
            old?.TrySetCanceled();
        }

        public static void TryCancelAndRemove<T>(ref TaskCompletionSource<T> completionSource, CancellationToken cancellationToken)
        {
            var old = completionSource;
            completionSource = null;
            old?.TrySetCanceled(cancellationToken);
        }

        public static void TrySetResultAndRemove<T>(ref UniTaskCompletionSource<T> completionSource, [CanBeNull] T result)
        {
            var old = completionSource;
            completionSource = null;
            old?.TrySetResult(result);
        }

        public static void TryCancelAndRemove<T>(ref UniTaskCompletionSource<T> completionSource, CancellationToken cancellationToken)
        {
            var old = completionSource;
            completionSource = null;
            old?.TrySetCanceled(cancellationToken);
        }

        public static void TryExceptionAndRemove<T>(ref UniTaskCompletionSource<T> completionSource, [NotNull] Exception exception)
        {
            var old = completionSource;
            completionSource = null;
            old?.TrySetException(exception);
        }

        public static void TrySetResultAndRemove_CancelAndReplaceToken<T>(
            ref UniTaskCompletionSource<T> completionSource,
            [CanBeNull] T result,
            ref CompositeCancellationTokenSource cts)
        {
            var oldCompletionSource = completionSource;
            var oldCts = cts;

            completionSource = null;
            cts = new CompositeCancellationTokenSource();

            oldCompletionSource?.TrySetResult(result);
            oldCts?.Cancel();
            oldCts?.Dispose();
        }

        public static void TryCancelAndRemove_CancelAndReplaceToken<T>(
            ref UniTaskCompletionSource<T> completionSource,
            CancellationToken cancellationToken,
            ref CompositeCancellationTokenSource cts)
        {
            var oldCompletionSource = completionSource;
            var oldCts = cts;

            completionSource = null;
            cts = new CompositeCancellationTokenSource();

            oldCompletionSource?.TrySetCanceled(cancellationToken);
            oldCts?.Cancel();
            oldCts?.Dispose();
        }

        public static void TryExceptionAndRemove_CancelAndReplaceToken<T>(
            ref UniTaskCompletionSource<T> completionSource,
            [NotNull] Exception exception,
            ref CompositeCancellationTokenSource cts)
        {
            var oldCompletionSource = completionSource;
            var oldCts = cts;

            completionSource = null;
            cts = new CompositeCancellationTokenSource();

            oldCompletionSource?.TrySetException(exception);
            oldCts?.Cancel();
            oldCts?.Dispose();
        }

        public static void CancelAndReplace(ref CompositeCancellationTokenSource cts)
        {
            var old = cts;
            cts = new();
            old?.Cancel();
            old?.Dispose();
        }

        public static void DiaposeAndReplace(ref CompositeCancellationTokenSource cts)
        {
            var old = cts;
            cts = new();
            old?.Dispose();
        }
    }
}