using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;

namespace Magify
{
    internal class TaskScheduler : MonoBehaviour
    {
        private readonly Queue<(Action Action, string Caller)> _tasksQueue = new Queue<(Action action, string Caller)>();
        private bool _dirty;

        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Scheduler);
        public static bool IsMainThread => PlayerLoopHelper.IsMainThread;

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static async UniTask SwitchToMainThread(CancellationToken cancellationToken)
        {
            if (!IsMainThread)
            {
                await UniTask.SwitchToMainThread(cancellationToken);
            }
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static async UniTask SwitchToThreadPool(CancellationToken cancellationToken)
        {
            await UniTask.SwitchToThreadPool();
            if (cancellationToken.IsCancellationRequested)
            {
                throw new OperationCanceledException(cancellationToken);
            }
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static async UniTask<T> RunOnMainThread<T>([NotNull] Func<T> function, CancellationToken cancellationToken)
        {
            if (!IsMainThread)
            {
                await UniTask.SwitchToMainThread(cancellationToken);
            }
            return function();
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static async UniTaskVoid RunOnMainThread([NotNull] Action function, CancellationToken cancellationToken)
        {
            if (!IsMainThread)
            {
                await UniTask.SwitchToMainThread(cancellationToken);
            }
            function();
        }

        public void Enqueue(Action action, string caller)
        {
            _logger.Log($"Scheduling {caller}");
            var execute = false;
            lock (_tasksQueue)
            {
                if (_tasksQueue.Count == 0 && IsMainThread)
                {
                    execute = true;
                }
                else
                {
                    _tasksQueue.Enqueue((action, caller));
                    _logger.Log($"Scheduled {caller}. Currently scheduled {_tasksQueue.Count}: {string.Join(", ", _tasksQueue.Select(c => c.Caller))}");
                    _dirty = true;
                }
            }
            if (execute)
            {
                _logger.Log($"Don't need to schedule {caller} because we are on main thread and no other tasks in queue");
                Execute(action, caller);
            }
        }

        public void ClearQueue()
        {
            lock (_tasksQueue)
            {
                _tasksQueue.Clear();
            }
        }

        private void Update()
        {
            if (!_dirty) return;
            _dirty = false;

            int count;
            lock (_tasksQueue)
            {
                count = _tasksQueue.Count;
                _logger.Log($"Executing scheduled tasks {count}: {string.Join(", ", _tasksQueue.Select(c => c.Caller))}");
            }

            while (count > 0)
            {
                Action action;
                string caller;
                lock (_tasksQueue)
                {
                    (action, caller) = _tasksQueue.Dequeue();
                    count = _tasksQueue.Count;
                }

                Execute(action, caller);
            }
        }

        private void Execute(Action action, string caller)
        {
            _logger.Log($"Trying to execute {caller}");
            try
            {
                action?.Invoke();
                _logger.Log($"Successfully called {caller}");
            }
            catch (Exception e)
            {
                _logger.LogException(e);
            }
        }
    }
}