using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Runtime.CompilerServices;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;

namespace Magify
{
    [CanBeNull]
    public delegate string GetCustomLogPrefixDelegate(UnityEngine.LogType logType, [CanBeNull] string tag);
}

namespace UnityEngine
{
    internal class MagifyLogger
    {
        [NotNull]
        private static readonly Dictionary<string, MagifyLogger> _loggers = new();
        [NotNull]
        private static readonly HashSet<string> _enabledTags = new(16);
        [NotNull]
        private static readonly MagifyLogger _defaultLogger = new(null);

        internal const string VerboseLoggingDefine = "MAGIFY_VERBOSE_LOGGING";

        private readonly string _tag;

        private static bool IsMainThread => PlayerLoopHelper.IsMainThread;

        public static bool IsLoggingEnabled { get; set; }

        public static Magify.GetCustomLogPrefixDelegate CustomPrefixSource { get; set; }

        private bool IsEnabled
        {
            [MethodImpl(MethodImplOptions.AggressiveInlining)]
            get => IsLoggingEnabled && (_tag == null || _enabledTags.Contains(_tag));
        }

        private MagifyLogger(string tag)
        {
            _tag = tag;
        }

        [NotNull]
        public static MagifyLogger Get(string tag = null)
        {
#if MAGIFY_VERBOSE_LOGGING
            if (string.IsNullOrEmpty(tag))
            {
                return _defaultLogger;
            }

            if (_loggers.TryGetValue(tag, out var logger) && logger != null)
            {
                return logger;
            }

            logger = new MagifyLogger(tag);
            _loggers[tag] = logger;

            return logger;
#else
            return _defaultLogger;
#endif
        }

        public static void EnableTag(string tag)
        {
            _enabledTags.Add(tag);
        }

        public static void DisableTag(string tag)
        {
            _enabledTags.Remove(tag);
        }

        public static bool IsTagActive(string tag)
        {
            return _enabledTags.Contains(tag);
        }

        public static IEnumerable<string> GetActiveTags()
        {
            return _enabledTags;
        }

        [Conditional(VerboseLoggingDefine)]
        public void Log(string message)
        {
            if (IsEnabled)
            {
                FormatAndLog(message, LogType.Log, Debug.Log);
            }
        }

        [Conditional(VerboseLoggingDefine)]
        public void Log(bool condition, string message)
        {
            if (condition && IsEnabled)
            {
                FormatAndLog(message, LogType.Log, Debug.Log);
            }
        }

        [Conditional(VerboseLoggingDefine)]
        public void Log(string message, Object context)
        {
            if (IsEnabled)
            {
                FormatAndLog(message, LogType.Log, context, Debug.Log);
            }
        }

        /// <param name="logType">Only <b>LogType.Log</b> | <b>LogType.Warning</b> | <b>LogType.Error</b> are possible</param>
        public void Log(string message, LogType logType)
        {
            switch (logType)
            {
                case LogType.Error:
                    LogError(message);
                    break;

                case LogType.Warning:
                    LogWarning(message);
                    break;

                case LogType.Log:
                default:
                    Log(message);
                    break;
            }
        }

        [Conditional(VerboseLoggingDefine)]
        public void LogWarning(string message)
        {
            if (IsEnabled)
            {
                FormatAndLog(message, LogType.Warning, Debug.LogWarning);
            }
        }

        [Conditional(VerboseLoggingDefine)]
        public void LogWarning(string message, Object context)
        {
            if (IsEnabled)
            {
                FormatAndLog(message, LogType.Warning, context, Debug.LogWarning);
            }
        }

        public void LogError(string message)
        {
            FormatAndLog(message, LogType.Error, Debug.LogError);
        }

        public void LogError(string message, Object context)
        {
            FormatAndLog(message, LogType.Error, context, Debug.LogError);
        }

        public void LogException(Exception exception)
        {
            Debug.LogException(exception);
        }

        private void FormatAndLog(string message, LogType type, [NotNull] Action<string> log)
        {
            if (IsMainThread)
            {
                log(string.IsNullOrEmpty(_tag)
                    ? $"[Magify][{type.AsString()}][{Time.frameCount}]{GetCustomPrefix(type)}: {message}"
                    : $"[Magify][{_tag}][{type.AsString()}][{Time.frameCount}]{GetCustomPrefix(type)}: {message}");
            }
            else
            {
                log(string.IsNullOrEmpty(_tag)
                    ? $"[Magify][{type.AsString()}]{GetCustomPrefix(type)}: {message}"
                    : $"[Magify][{_tag}][{type.AsString()}]{GetCustomPrefix(type)}: {message}");
            }
        }

        private void FormatAndLog(string message, LogType type, Object context, [NotNull] Action<string, Object> log)
        {
            if (IsMainThread)
            {
                log(string.IsNullOrEmpty(_tag)
                    ? $"[Magify][{type.AsString()}][{Time.frameCount}]{GetCustomPrefix(type)}: {message}"
                    : $"[Magify][{_tag}][{type.AsString()}][{Time.frameCount}]{GetCustomPrefix(type)}: {message}", context);
            }
            else
            {
                log(string.IsNullOrEmpty(_tag)
                    ? $"[Magify][{type.AsString()}]{GetCustomPrefix(type)}: {message}"
                    : $"[Magify][{_tag}][{type.AsString()}]{GetCustomPrefix(type)}: {message}", context);
            }
        }

        private string GetCustomPrefix(LogType logType)
        {
            var prefix = CustomPrefixSource?.Invoke(logType, _tag);
            return prefix is null ? null : $"[{prefix}]";
        }

        public static void Reset()
        {
            _enabledTags.Clear();
            CustomPrefixSource = null;
        }
    }
}