﻿using System;
using System.Linq;
using System.Text.RegularExpressions;

namespace Magify
{
    internal static class EmailValidator
    {
        public enum EmailValidationError
        {
            Unexpected,
            Empty,
            TooLong,
            NoAtSign,
            DuplicatedAtSign,
        }

        public static bool IsValidEmail(string email, out EmailValidationError error)
        {
            error = EmailValidationError.Unexpected;
            if (string.IsNullOrWhiteSpace(email))
            {
                error = EmailValidationError.Empty;
                return false;
            }
            if (email.Length > 254)
            {
                error = EmailValidationError.TooLong;
                return false;
            }
            var atSignNumber = email.Count(s => s == '@');
            switch (atSignNumber)
            {
                case 0:
                    error = EmailValidationError.NoAtSign;
                    return false;
                case > 1:
                    error = EmailValidationError.DuplicatedAtSign;
                    return false;
            }

            try
            {
                return Regex.IsMatch(
                    email,
                    @"^(?=.{1,254}$)(?=.{1,64}@.{1,253}$)[^@\s]+@[^@\s]+\.[^@\s]{2,63}$",
                    RegexOptions.IgnoreCase,
                    TimeSpan.FromMilliseconds(250));
            }
            catch (RegexMatchTimeoutException)
            {
                return false;
            }
        }
    }
}