﻿using System;
using System.Collections.Generic;
using System.IO;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    internal class TypedSection<T> : Section<T, T>, IDisposable
    {
        [NotNull]
        private readonly Dictionary<string, ReactiveProperty<T>> _reactiveProperties = new(30);
        [NotNull]
        private readonly Dictionary<ReactiveProperty<T>, IDisposable> _disposables = new(30);

        public override event Action OnChanged;

        [NotNull]
        public override string Name => BinaryType.TypeName;

        public TypedSection([NotNull] BinaryTypeHandler<T> binaryType) : base(binaryType)
        {
        }

        [NotNull]
        public IReactiveProperty<T> GetReactiveProperty([NotNull] string key, T initValue = default)
        {
            if (!_reactiveProperties.TryGetValue(key, out var prop) || prop == null)
            {
                var value = Get(key, initValue);
                prop = new ReactiveProperty<T>(value);
                _reactiveProperties[key] = prop;
                _disposables[prop] = prop
                    .SkipLatestValueOnSubscribe()
                    .Subscribe(v => { Set(key, v); });
            }
            return prop;
        }

        public bool Has([NotNull] string key)
        {
            return Data.ContainsKey(key);
        }

        public T Get([NotNull] string key, T initValue = default)
        {
            var res = TryGet(key);
            if (res.Exists) return res.Value;
            Data[key] = initValue;
            OnChanged?.Invoke();
            return initValue;
        }

        public bool Set([NotNull] string key, T value)
        {
            var res = TryGet(key);
            if (res.Exists && Equals(res.Value, value))
            {
                return false;
            }

            Data[key] = value;
            OnChanged?.Invoke();

            if (_reactiveProperties.TryGetValue(key, out var prop) && prop != null)
            {
                prop.Value = value;
            }

            return true;
        }

        public override bool Remove([NotNull] string key)
        {
            var removed = Data.Remove(key);
            if (removed)
            {
                OnChanged?.Invoke();
            }
            if (_reactiveProperties.TryGetValue(key, out var prop) && prop != null)
            {
                prop.Dispose();
                prop.Value = default;
                _disposables[prop]?.Dispose();
                _disposables.Remove(prop);
            }
            return removed;
        }

        public override void Reset()
        {
            if (Data.Count <= 0) return;
            Data.Clear();
            OnChanged?.Invoke();
            ClearReactiveProperties();
        }

        private void ClearReactiveProperties()
        {
            foreach (var value in _disposables.Values)
            {
                value?.Dispose();
            }
            foreach (var pair in _reactiveProperties)
            {
                pair.Value?.Dispose();
            }
            _reactiveProperties.Clear();
            _disposables.Clear();
        }

        public override void WriteTo([NotNull] BinaryWriter writer)
        {
            writer.Write(Data.Count);
            foreach (var pair in Data)
            {
                writer.Write(pair.Key!);
                writer.Write(BinaryType.SizeOf(pair.Value));
                BinaryType.WriteTo(writer, pair.Value);
            }
        }

        public override void ReadFrom([NotNull] BinaryReader reader)
        {
            var count = reader.ReadInt32();
            for (var i = 0; i < count; i++)
            {
                var key = reader.ReadString();
                reader.ReadInt32(); // skip reading size of value
                var value = BinaryType.ReadFrom(reader);
                Data[key] = value;
                if (_reactiveProperties.TryGetValue(key, out var prop) && prop != null && !Equals(prop.Value, value))
                    prop.Value = value;
            }
        }

        void IDisposable.Dispose()
        {
            ClearReactiveProperties();
        }
    }
}