﻿using System;
using System.Collections;
using System.Collections.Generic;
using JetBrains.Annotations;

namespace Magify
{
    public class ReactiveList<T> : IReactiveCollection, IRefillableCollection<T>, IList<T>, IReadOnlyList<T>
    {
        [NotNull]
        private readonly List<T> _list = new List<T>();

        public event Action OnChanged;

        private void SetDirty()
        {
            OnChanged?.Invoke();
        }

        #region Mutable functionallity

        public T this[int index]
        {
            get => _list[index];
            set
            {
                _list[index] = value;
                SetDirty();
            }
        }

        public void Add(T item)
        {
            _list.Add(item);
            SetDirty();
        }

        public void Clear()
        {
            if (_list.Count > 0)
            {
                _list.Clear();
                SetDirty();
            }
        }

        public bool Remove(T item)
        {
            var removed = _list.Remove(item);
            if (removed)
            {
                SetDirty();
            }
            return removed;
        }

        public void Insert(int index, T item)
        {
            _list.Insert(index, item);
            SetDirty();
        }

        public void RemoveAt(int index)
        {
            _list.RemoveAt(index);
            SetDirty();
        }

        public void Refill(IEnumerable<T> elements)
        {
            var changed = _list.Count;
            _list.Clear();
            foreach (var element in elements)
            {
                changed++;
                _list.Add(element);
            }
            if (changed > 0)
                SetDirty();
        }

        #endregion

        #region Immutable functionallity

        public int Count => _list.Count;

        public bool IsReadOnly => ((IList<T>)_list).IsReadOnly;

        public int IndexOf(T item) => _list.IndexOf(item);

        public IEnumerator<T> GetEnumerator() => _list.GetEnumerator();

        IEnumerator IEnumerable.GetEnumerator() => _list.GetEnumerator();

        public bool Contains(T item) => _list.Contains(item);

        public void CopyTo(T[] array, int arrayIndex) => _list.CopyTo(array, arrayIndex);

        #endregion
    }
}