﻿using System;
using System.Collections;
using System.Collections.Generic;
using JetBrains.Annotations;

namespace Magify
{
    internal class ReactiveDictionary<TKey, TValue> : IReactiveCollection, IRefillableCollection<KeyValuePair<TKey, TValue>>, IDictionary<TKey, TValue>, IReadOnlyDictionary<TKey, TValue>
    {
        [NotNull]
        private readonly Dictionary<TKey, TValue> _dictionary = new();

        public event Action OnChanged;

        private void SetDirty()
        {
            OnChanged?.Invoke();
        }

        #region Mutable functionallity

        public TValue this[TKey key]
        {
            get => _dictionary[key];
            set
            {
                _dictionary[key] = value;
                SetDirty();
            }
        }

        public void Add(KeyValuePair<TKey, TValue> item)
        {
            ((ICollection<KeyValuePair<TKey, TValue>>)_dictionary).Add(item);
            SetDirty();
        }

        public void Clear()
        {
            _dictionary.Clear();
            SetDirty();
        }

        public bool Remove(KeyValuePair<TKey, TValue> item)
        {
            var removed = ((ICollection<KeyValuePair<TKey, TValue>>)_dictionary).Remove(item);
            if (removed)
            {
                SetDirty();
            }
            return removed;
        }

        public void Add(TKey key, TValue value)
        {
            _dictionary.Add(key, value);
            SetDirty();
        }

        public bool Remove(TKey key)
        {
            var removed = _dictionary.Remove(key);
            if (removed)
            {
                SetDirty();
            }
            return removed;
        }

        public void Refill([CanBeNull] IEnumerable<KeyValuePair<TKey, TValue>> elements)
        {
            var changed = _dictionary.Count;
            _dictionary.Clear();
            changed += _dictionary.AddRange(elements);
            if (changed > 0)
                SetDirty();
        }

        #endregion

        #region Immutable functionallity

        public int Count => _dictionary.Count;

        public bool IsReadOnly => ((IDictionary<TKey, TValue>)_dictionary).IsReadOnly;

        public ICollection<TKey> Keys => _dictionary.Keys;

        public ICollection<TValue> Values => _dictionary.Values;

        public IEnumerator<KeyValuePair<TKey, TValue>> GetEnumerator() => _dictionary.GetEnumerator();

        IEnumerable<TKey> IReadOnlyDictionary<TKey, TValue>.Keys => Keys;

        IEnumerable<TValue> IReadOnlyDictionary<TKey, TValue>.Values => Values;

        IEnumerator IEnumerable.GetEnumerator() => GetEnumerator();

        public bool Contains(KeyValuePair<TKey, TValue> item) => ((ICollection<KeyValuePair<TKey, TValue>>)_dictionary).Contains(item);

        public void CopyTo(KeyValuePair<TKey, TValue>[] array, int arrayIndex) => ((ICollection<KeyValuePair<TKey, TValue>>)_dictionary).CopyTo(array, arrayIndex);

        public bool ContainsKey(TKey key) => _dictionary.ContainsKey(key);

        public bool TryGetValue(TKey key, out TValue value) => _dictionary.TryGetValue(key, out value);

        #endregion
    }
}