﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using JetBrains.Annotations;
using Magify.Rx;
using Newtonsoft.Json;

namespace Magify
{
    public partial class BinaryStorage
    {
#if UNITY_EDITOR
        [NotNull]
        private static readonly HashSet<string> _lockedFiles = new();
#endif
#if MAGIFY_VERBOSE_LOGGING
        [NotNull]
        private static readonly Dictionary<string, object> _dictionary = new();
        [NotNull]
        private static readonly Dictionary<Type, PropertyInfo> _typeToDataGetter = new();
        [NotNull]
        private static readonly Subject<(string Name, string Json)> _onStorageSavedAsJson = new();
        [NotNull]
        internal static IObservable<(string Name, string Json)> OnStorageSavedAsJson => _onStorageSavedAsJson;
#endif

        static partial void SaveAsJsonForDebug(string filePath, [NotNull, ItemNotNull] List<Section> sections)
        {
#if MAGIFY_VERBOSE_LOGGING
            try
            {
                _dictionary.Clear();
                foreach (var section in sections.Where(c => c!.HasData))
                {
                    var sectionType = section!.GetType();
                    if (!_typeToDataGetter.TryGetValue(sectionType, out var getter))
                    {
                        getter = section.GetType().GetProperty("Data", BindingFlags.Instance | BindingFlags.NonPublic);
                        _typeToDataGetter[sectionType] = getter;
                    }
                    if (getter == null)
                    {
                        continue;
                    }
                    _dictionary[section.Name] = getter.GetMethod!.Invoke(section, Array.Empty<object>());
                }
                var content = JsonConvert.SerializeObject(_dictionary, Formatting.Indented);
                File.WriteAllText(filePath + ".json", content);
                _onStorageSavedAsJson.OnNext((Path.GetFileNameWithoutExtension(filePath), content));
            }
            catch
            {
                // ignored
            }
#endif
        }

#if UNITY_EDITOR
        static partial void ThrowIfFilePathLocked([NotNull] string filePath)
        {
            filePath = Path.GetFullPath(filePath).TrimEnd('\\');
            if (_lockedFiles.Any(s => filePath == s))
            {
                throw new MagifyBinaryStorageLockedException(filePath);
            }
        }

        static partial void LockFilePathInEditor([NotNull] string filePath)
        {
            filePath = Path.GetFullPath(filePath).TrimEnd('\\');
            _lockedFiles.Add(filePath);
        }

        static partial void UnlockFilePathInEditor([NotNull] string filePath)
        {
            filePath = Path.GetFullPath(filePath).TrimEnd('\\');
            _lockedFiles.Remove(filePath);
        }
#endif
    }
}
