using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;
using Magify.Model;

namespace Magify
{
    internal readonly struct MagifyContextSyncJob : IMagifyWorkerJob
    {
        public ConfigScope Scopes { get; }
        public CancellationToken CancellationToken { get; }

        public MagifyContextSyncJob(ConfigScope scopes, CancellationToken cancellationToken)
        {
            Scopes = scopes;
            CancellationToken = cancellationToken;
        }
    }

    internal class MagifyContextSyncWorker : IMagifyWorker<MagifyContextSyncJob>
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        [NotNull]
        private readonly Action _onRequestedCallback;
        [NotNull]
        private readonly Action<ContextSyncTime> _onSyncedCallback;
        [NotNull]
        private readonly IServerApi _serverApi;
        [NotNull]
        private readonly ContextApplicator _contextApplicator;
        [NotNull]
        private readonly ContextFileStorage _contextFileStorage;

        public MagifyContextSyncWorker(
            [NotNull] Action onRequestedCallback,
            [NotNull] Action<ContextSyncTime> onSyncedCallback,
            [NotNull] IServerApi serverApi,
            [NotNull] ContextApplicator contextApplicator,
            [NotNull] ContextFileStorage contextFileStorage)
        {
            _onRequestedCallback = onRequestedCallback;
            _onSyncedCallback = onSyncedCallback;
            _serverApi = serverApi;
            _contextApplicator = contextApplicator;
            _contextFileStorage = contextFileStorage;
        }

        public void DoJob([NotNull] MagifyContextSyncJob job)
        {
            SyncContext(job.Scopes, job.CancellationToken).Forget();
        }

        private async UniTaskVoid SyncContext(ConfigScope scopes, CancellationToken cancellationToken)
        {
            _onRequestedCallback();
            _logger.Log($"{nameof(SyncContext)} has been called");
            if (_serverApi.IsContextRequest)
            {
                _logger.Log($"{nameof(SyncContext)} skipped because context already requested");
                return;
            }

            CampaignsContext context;
            try
            {
                await TaskScheduler.SwitchToThreadPool(cancellationToken);
                context = await _serverApi.GetContextAsync(scopes, cancellationToken);
                await TaskScheduler.SwitchToMainThread(cancellationToken);
                if (context == null)
                {
                    _logger.Log($"{nameof(SyncContext)} finished with context: null");
                    return;
                }
            }
            catch (OperationCanceledException)
            {
                _logger.Log($"Loading of context with scopes {scopes} was cancelled");
                return;
            }
            catch (Exception e)
            {
                _logger.LogException(e);
                return;
            }

            _logger.Log($"Load context: ContextBuildTime={context.ContextBuildTime}");
            _onSyncedCallback(new ContextSyncTime(DateTime.UtcNow.ToUnixMilliseconds(), context.ContextBuildTime?.ToUnixMilliseconds() ?? 0));

            UniTask.RunOnThreadPool(() => SaveContext(context)).Forget();
            _contextApplicator.HandleContext(context, ContextKind.Downloaded);
        }

        private void SaveContext([NotNull] CampaignsContext context)
        {
            var json = _contextFileStorage.LoadContext();
            CampaignsContext savedContext;
            if (string.IsNullOrEmpty(json))
            {
                savedContext = context;
            }
            else
            {
                savedContext = CampaignsContext.Deserialize(json)!;
                savedContext.PullChangesFrom(context);
            }

            json = JsonFacade.SerializeObject(savedContext);
            _contextFileStorage.SaveContext(json);
        }
    }
}