using System;
using UnityEngine;
using Object = UnityEngine.Object;

namespace Magify
{
    public class ResourcesUtils
    {
        public static DisposableInstance<T> Instantiate<T>(string url)
            where T : MonoBehaviour
        {
            var resource = Load<T>(url);
            return resource == null ? null : new DisposableInstance<T>(resource);
        }

        public static DisposableResource<T> Load<T>(string url)
            where T : MonoBehaviour
        {
            var load = Resources.Load<T>(url);
            return load == null ? null : new DisposableResource<T>(load);
        }
    }

    public class DisposableResource<T> : IDisposable
        where T : MonoBehaviour
    {
        private bool _isDisposed;

        public T Asset { get; }

        public DisposableResource(T asset)
        {
            Asset = asset;
        }

        public void Dispose()
        {
            if (!_isDisposed)
            {
                _isDisposed = true;
                Resources.UnloadAsset(Asset);
            }
        }
    }

    public class DisposableInstance<T> : IDisposable
        where T : MonoBehaviour
    {
        private bool _isDisposed;

        public T Instance { get; }

        public DisposableInstance(DisposableResource<T> resource)
        {
            Instance = Object.Instantiate(resource.Asset);
        }

        public void Dispose()
        {
            if (!_isDisposed)
            {
                _isDisposed = true;
                Object.Destroy(Instance.gameObject);
            }
        }
    }
}