﻿using System;
using System.Linq;
using System.Text;
using JetBrains.Annotations;
using Magify.Model;

namespace Magify
{
    internal static class JwtDecoder
    {
        [NotNull]
        public static JwtPayload DecodePayload([NotNull] string token)
        {
            var parts = SplitToken(token);
            return JsonFacade.DeserializeObject<JwtPayload>(Base64Decode(parts[1]))!;
        }

        [NotNull]
        [ItemNotNull]
        private static string[] SplitToken([NotNull] string token)
        {
            var parts = token.Split('.')!.Where(p => !string.IsNullOrWhiteSpace(p)).ToArray();
            if (parts.Length == 2 && token.EndsWith("."))
            {
                // Tokens with alg='none' have empty String as Signature.
                parts = new[] { parts[0], parts[1], "" };
            }
            if (parts.Length != 3)
            {
                throw new TooShortMagifyTokenException(parts.Length);
            }
            return parts;
        }

        [NotNull]
        private static string Base64Decode([NotNull] string base64String)
        {
            try
            {
                var bytes = Convert.FromBase64String(PadBase64String(base64String));
                return Encoding.UTF8.GetString(bytes);
            }
            catch (FormatException e)
            {
                throw new DecodeException("Received bytes didn't correspond to a valid Base64 encoded string.", e);
            }
        }

        [NotNull]
        private static string PadBase64String([NotNull] string str)
        {
            // When encoding or decoding Base64, strings are often "padded" with = characters to make their length a multiple of 4.
            // This padding is essential because Base64 encoding represents binary data in a 4-character set, so the string length must be divisible by 4 for proper decoding.
            // Convert.FromBase64String() requires that the input string be padded to a multiple of 4 characters, but we receive shorter one.
            // This code ensures that the string has this correct length by adding any necessary padding chars.
            return str.PadRight(str.Length + (4 - str.Length % 4) % 4, '=');
        }
    }
}