using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class PlatformDefault : PlatformAPI
    {
        private const string PlayerPrefsPrefix = "magify_editor_test_prefs";
        private const string PlayerPrefsSavedKeys = "magify_editor_test_prefs_saved_keys";
        private const string CloudPrefix = "magify_editor_test_cloud";
        private const string CloudSavedKeys = "magify_editor_test_cloud_saved_keys";

        [CanBeNull]
        private readonly EditorDevice _device;

        private readonly IDisposable _disposable;
        [NotNull]
        private readonly CancellationTokenSource _lifetimeCts = new();
        [NotNull]
        private readonly Subject<object> _onBackground = new();
        [NotNull]
        private readonly Subject<object> _onForeground = new();
        [NotNull]
        private readonly Subject<LocaleData> _onLocaleChanged = new();
        [NotNull]
        private readonly Subject<string> _onNewPackageInstalled = new();

        [NotNull]
        private readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Editor);
        [NotNull]
        private readonly HashSet<string> _installedAppIdntifiers = new();
        [NotNull]
        private LocaleData _locale = new();

        public override RuntimePlatform RuntimePlatform => _device?.Platform ?? Application.platform;
        [NotNull]
        public override IObservable<object> OnBackground => _onBackground;
        [NotNull]
        public override IObservable<object> OnForeground => _onForeground;
        [NotNull]
        public override IObservable<LocaleData> OnLocaleChanged => _onLocaleChanged;
        [NotNull]
        public override IObservable<string> OnNewPackageInstalled => _onNewPackageInstalled;

        public override string DeviceName => _device?.DeviceName ?? "phone-xhdpi";
        public override string DeviceVersion => _device?.DeviceVersion ?? "30";
        public override string BuildNumber => CustomBuildNumber ?? _device?.BuildNumber;
        public override bool IsTablet => _device?.IsTablet ?? base.IsTablet;
        public override Orientation SupportedOrientation => _device?.SupportedOrientation ?? base.SupportedOrientation;

        public PlatformDefault(EditorDevice device = null)
        {
            _device = device;
            CustomGpsAdId = _device?.GpsAdId;
            CustomIdfa = _device?.Idfa;
            CustomIdfv = _device?.Idfv;

            if (Application.isEditor && !Application.isPlaying)
            {
                return;
            }
            _disposable = Observable.EveryApplicationFocus().Subscribe(focus =>
            {
                if (!focus)
                {
                    _onBackground.OnNext(this);
                }
                else
                {
                    _onForeground.OnNext(this);
                }
            });
        }

        public override void AddPlatformSpecificData(AuthTokenPayload authToken, GeneralPrefs prefs, bool isGeoIpEnabled, bool syncStateEnabled)
        {
            base.AddPlatformSpecificData(authToken, prefs, isGeoIpEnabled, syncStateEnabled);
            if (_device?.Platform is RuntimePlatform.Android or RuntimePlatform.IPhonePlayer)
            {
                authToken.IsGeoIPEnabled = isGeoIpEnabled ? null : false;
            }
            if (_device?.Platform is RuntimePlatform.Android)
            {
                authToken.GpsAdId = prefs.GpsAdId.Value;
                authToken.AndroidId = syncStateEnabled ? CustomAndroidId ?? GetAndroidId() : null;
            }
            if (_device?.Platform is RuntimePlatform.IPhonePlayer)
            {
                authToken.IsIdfaAvailable = prefs.AttAuthorizationStatus.Value;
                authToken.Idfa = GetIdfa();
                authToken.Idfv = GetIdfv();
            }
        }

        public void GoToBackground()
        {
            _onBackground.OnNext(this);
        }

        public void ReturnToForeground()
        {
            _onForeground.OnNext(this);
        }

        [NotNull]
        public override LocaleData GetCurrentLocale()
        {
            return _locale;
        }

        public override void Dispose()
        {
            _disposable?.Dispose();
            _lifetimeCts.Cancel();
            _lifetimeCts.Dispose();
            base.Dispose();
        }

        [NotNull]
        public override string GetAndroidId()
        {
            return _device?.AndroidId ?? "00000000-just-f00r-test-000000000000";
        }

        public override void TweakUserLocale(string languageTag)
        {
            _locale = new LocaleData("US", languageTag);
            _onLocaleChanged.OnNext(_locale);
        }

        public override void TrackPromotedApplication(string bundleId)
        {
            if (bundleId == null) return;
            _installedAppIdntifiers.Add(bundleId);
            _onNewPackageInstalled.OnNext(bundleId);
        }

        public override bool IsApplicationInstalled(string identifier)
        {
            return _installedAppIdntifiers.Contains(identifier);
        }

        public override bool IsApplicationInstalled(PromotedApplication promotedApplication)
        {
            return _device?.Platform switch
            {
                RuntimePlatform.IPhonePlayer => promotedApplication?.Schemas?.Any(IsApplicationInstalled) ?? false,
                RuntimePlatform.Android or _ => IsApplicationInstalled(promotedApplication?.BundleId),
            };
        }

        public override void SetValueInNativeStorage(string value, string key)
        {
            try
            {
                TaskScheduler.RunOnMainThread(() =>
                {
                    PlayerPrefs.SetString(value, PlayerPrefsPrefix + key);
                    var keys = JsonFacade.DeserializeObject<HashSet<string>>(PlayerPrefs.GetString(PlayerPrefsSavedKeys, string.Empty) ?? string.Empty);
                    keys ??= new HashSet<string>();
                    keys.Add(PlayerPrefsPrefix + key);
                    PlayerPrefs.SetString(PlayerPrefsSavedKeys, JsonFacade.SerializeObject(keys));
                }, _lifetimeCts.Token).Forget();
            }
            catch (Exception e)
            {
                _logger.LogWarning($"Failed to set value ({value}) to fake native storage for key {key}: {e.Message}");
            }
        }

        public override string GetValueInNativeStorage(string key)
        {
            try
            {
                return PlayerPrefs.GetString(PlayerPrefsPrefix + key, null);
            }
            catch (Exception e)
            {
                _logger.LogWarning($"Failed to get value from fake native storage for key {key}: {e.Message}");
                return null;
            }
        }

        public override void RemoveValueInNativeStorage(string key)
        {
            try
            {
                TaskScheduler.RunOnMainThread(() =>
                {
                    PlayerPrefs.DeleteKey(PlayerPrefsPrefix + key);
                }, _lifetimeCts.Token).Forget();
            }
            catch (Exception e)
            {
                _logger.LogWarning($"Failed to remove value from fake native storage for key {key}: {e.Message}");
            }
        }

        public override void ClearNativeStorage()
        {
            try
            {
                TaskScheduler.RunOnMainThread(() =>
                {
                    var keys = JsonFacade.DeserializeObject<HashSet<string>>(PlayerPrefs.GetString(PlayerPrefsSavedKeys, string.Empty) ?? string.Empty);
                    foreach (var key in keys ?? Enumerable.Empty<string>())
                    {
                        RemoveValueInNativeStorage(key);
                    }
                    PlayerPrefs.DeleteKey(PlayerPrefsSavedKeys);
                }, _lifetimeCts.Token).Forget();
            }
            catch (Exception e)
            {
                _logger.LogWarning($"Failed to clear fake native storage: {e.Message}");
            }
        }

        public override void SetValueInCloudStorage(string key, string value)
        {
            try
            {
                TaskScheduler.RunOnMainThread(() =>
                {
                    PlayerPrefs.SetString(value, CloudPrefix + key);
                    var keys = JsonFacade.DeserializeObject<HashSet<string>>(PlayerPrefs.GetString(CloudSavedKeys, string.Empty) ?? string.Empty);
                    keys ??= new HashSet<string>();
                    keys.Add(CloudPrefix + key);
                    PlayerPrefs.SetString(CloudSavedKeys, JsonFacade.SerializeObject(keys));
                }, _lifetimeCts.Token).Forget();
            }
            catch (Exception e)
            {
                _logger.LogWarning($"Failed to set value ({value}) to fake native cloud storage for key {key}: {e.Message}");
            }
        }

        public override async UniTask<string> GetValueFromCloudStorage(string key, double timeoutSeconds, CancellationToken cancellationToken)
        {
            try
            {
                await TaskScheduler.SwitchToMainThread(cancellationToken);
                return PlayerPrefs.GetString(CloudPrefix + key, null);
            }
            catch (Exception e)
            {
                _logger.LogWarning($"Failed to get value from fake native cloud storage for key {key}: {e.Message}");
                return null;
            }
        }

        public override void RemoveValueFromCloudStorage([NotNull] string key)
        {
            try
            {
                TaskScheduler.RunOnMainThread(() => { PlayerPrefs.DeleteKey(CloudPrefix + key); }, _lifetimeCts.Token).Forget();
            }
            catch (Exception e)
            {
                _logger.LogWarning($"Failed to remove value from fake native cloud storage for key {key}: {e.Message}");
            }
        }
    }
}