using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    /// <summary>
    /// Platform specific tools and info
    /// </summary>
    internal abstract class PlatformAPI : ICancelable
    {
        public bool IsDisposed { get; protected set; }
        public abstract RuntimePlatform RuntimePlatform { get; }
        public abstract IObservable<object> OnBackground { get; }
        public abstract IObservable<object> OnForeground { get; }
        [CanBeNull]
        public virtual IObservable<object> OnApplicationDidBecomeActive => null;
        [CanBeNull]
        public virtual IObservable<object> OnApplicationWillResignActive => null;
        public abstract IObservable<LocaleData> OnLocaleChanged { get; }
        public abstract IObservable<string> OnNewPackageInstalled { get; }

        public abstract string DeviceName { get; }
        public abstract string DeviceVersion { get; }
        public virtual string BuildNumber { get; }

        [CanBeNull]
        internal string CustomIdfa { get; set; }

        [CanBeNull]
        internal string CustomIdfv { get; set; }

        [CanBeNull]
        internal string CustomGpsAdId { get; set; }

        [CanBeNull]
        internal string CustomAndroidId { get; set; }

        [CanBeNull]
        internal string CustomBuildNumber { get; set; }

        [CanBeNull]
        internal MigrationData CustomMigrationData { get; set; }

        [NotNull]
        public static PlatformAPI PreparePlatformBridge(EditorDevice editorDevice, string rootPath)
        {
            return
#if !UNITY_EDITOR && UNITY_ANDROID
                new PlatformAndroid(rootPath);
#elif !UNITY_EDITOR && UNITY_IOS
                new PlatformIOS();
#else
                new PlatformDefault(editorDevice);
#endif
        }

        public abstract void TweakUserLocale(string languageTag);
        public abstract void TrackPromotedApplication(string identifier);
        public abstract bool IsApplicationInstalled([CanBeNull] string identifier);
        public abstract bool IsApplicationInstalled([CanBeNull] PromotedApplication application);

        public virtual Orientation SupportedOrientation
        {
            get
            {
                var allowPortrait = Screen.autorotateToPortrait || Screen.autorotateToPortraitUpsideDown;
                var allowLandscape = Screen.autorotateToLandscapeLeft || Screen.autorotateToLandscapeRight;
                if (allowPortrait && allowLandscape)
                {
                    return Orientation.Both;
                }
                return allowLandscape ? Orientation.Landscape : Orientation.Portrait;
            }
        }

        public virtual bool IsTablet
        {
            get
            {
                var max = Mathf.Max((float)Screen.width, Screen.height);
                var min = Mathf.Min((float)Screen.width, Screen.height);
                var aspectRatio = max / min;
                return DeviceDiagonalSizeInInches() > 6.5f && aspectRatio < 2f;
            }
        }

        public virtual MigrationData LoadLegacyNativeDataForMigration()
        {
            return CustomMigrationData;
        }

        public virtual void CleanLegacyNativeDataAfterMigration()
        {
        }

        public virtual int? GetGlobalCounterFromLegacyData(CounterType type, CounterKey key)
        {
            return null;
        }

        private static float DeviceDiagonalSizeInInches()
        {
            var screenWidth = Screen.width / Screen.dpi;
            var screenHeight = Screen.height / Screen.dpi;
            var diagonalInches = Mathf.Sqrt(Mathf.Pow(screenWidth, 2) + Mathf.Pow(screenHeight, 2));
            return diagonalInches;
        }

        [NotNull]
        public abstract LocaleData GetCurrentLocale();

        public virtual void AddPlatformSpecificData([NotNull] AuthTokenPayload authToken, [NotNull] GeneralPrefs prefs, bool isGeoIpEnabled, bool syncStateEnabled)
        {
        }

        public virtual void Dispose()
        {
            IsDisposed = true;
        }

        public virtual void SetupConversionTracker(string revenuePerCountryPath, string revenueLevelsPath, string defaultCurrencyRatesPath, string subscriptionMultipliersPath)
        {
        }

        public virtual string GetIdfa()
        {
            return CustomIdfa;
        }

        public virtual string GetIdfv()
        {
            return CustomIdfv;
        }

        public virtual string GetEncodedDeviceIdentifier()
        {
            return null;
		}

        public virtual string GetStoreCountry()
        {
            return default;
        }

        [CanBeNull]
        public virtual string GetGpsAdId()
        {
            return CustomGpsAdId;
        }

        [CanBeNull]
        public virtual string GetAndroidId()
        {
            return null;
        }

        public virtual string GetValueInNativeStorage(string key)
        {
            return null;
        }

        public virtual void SetValueInNativeStorage(string value, string key)
        {
        }

        public virtual void RemoveValueInNativeStorage(string key)
        {
        }

        /// <summary>
        /// iOS - clears all keychains <br/>
        /// Android - does nothing
        /// </summary>
        public virtual void ClearNativeStorage()
        {
        }

        public virtual UniTask<string> GetValueFromCloudStorage([NotNull] string key, double timeoutSeconds, CancellationToken cancellationToken)
        {
            return UniTask.FromResult<string>(default);
        }

        public virtual void SetValueInCloudStorage([NotNull] string key, [NotNull] string value)
        {
        }

        public virtual void RemoveValueFromCloudStorage([NotNull] string key)
        {
        }
    }
}