using System;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;

namespace Magify
{
    public enum ServerInteractionTaskCreationStatus
    {
        SuccessfullyCreated = 0,
        OfflineModeIsEnabled = 1,
        NoCompatibleModeFound = 2,
    }

    internal readonly struct ServerInteractionTask<TOut>
    {
        [CanBeNull]
        public UniTask<TOut>? Value { get; init; }
        public ServerInteractionTaskCreationStatus CreationStatus { get; init; }

        public static ServerInteractionTask<TOut> SuccessfullyCreated(UniTask<TOut> value)
        {
            return new ServerInteractionTask<TOut>
            {
                Value = value,
                CreationStatus = ServerInteractionTaskCreationStatus.SuccessfullyCreated,
            };
        }

        public static ServerInteractionTask<TOut> OfflineModeIsEnabled()
        {
            return new ServerInteractionTask<TOut>
            {
                Value = null,
                CreationStatus = ServerInteractionTaskCreationStatus.OfflineModeIsEnabled,
            };
        }

        public static ServerInteractionTask<TOut> NoCompatibleModeFound()
        {
            return new ServerInteractionTask<TOut>
            {
                Value = null,
                CreationStatus = ServerInteractionTaskCreationStatus.NoCompatibleModeFound,
            };
        }

        public UniTask<TOut> HandleByDefault(RequestConfig requestConfig, TOut rejectedResult)
        {
            return CreationStatus switch
            {
                ServerInteractionTaskCreationStatus.SuccessfullyCreated => Value!.Value,
                ServerInteractionTaskCreationStatus.OfflineModeIsEnabled => UniTask.FromResult(rejectedResult),
                ServerInteractionTaskCreationStatus.NoCompatibleModeFound => throw new MagifyServerInteractionModeNotFoundException(requestConfig),
                _ => throw new ArgumentOutOfRangeException()
            };
        }
    }

    internal static class ServerInteractionTaskExtensions
    {
        public static ServerInteractionTask<TOut> ToServerInteractionTask<TOut>(this UniTask<TOut> value)
        {
            return ServerInteractionTask<TOut>.SuccessfullyCreated(value);
        }
    }
}