using System;
using System.Threading;
using System.Collections.Generic;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal delegate RepeatState ResponseProcessorDelegate<TOut>(WebResponseMessage webResponseMessage, out TOut result);
    internal delegate CancellationToken ActiveRequestCancellationTokenProviderDelegate();
    [ItemNotNull]
    internal delegate UniTask<string> GetAuthTokenAsyncDelegate(CancellationToken cancellationToken);

    internal partial class ServerApi : IServerApi, IInitializable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Network);
        [NotNull]
        private readonly GeneralPrefs _generalPrefs;
        [NotNull]
        private readonly AppStatePrefs _appStatePrefs;
        [NotNull]
        private readonly IAuthorizationConfigBuilder _authConfigBuilder;
        [NotNull]
        private readonly PlatformAPI _platform;
        [NotNull]
        private readonly ServerInteractionsProvider _interactionsProvider;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        [NotNull, ItemNotNull]
        private readonly List<ICancelable> _disposableDependencies;
        [NotNull]
        private readonly RequestContext<string> _requestNewTokenContext;
        [NotNull]
        private readonly RequestContext<CampaignsContext> _requestContextContext;
        [NotNull]
        private readonly RequestContext<bool> _requestStoreEventsContext;
        [NotNull]
        private readonly RequestContext<PurchaseVerificationResult> _requestVerifyPurchaseContext;
        [NotNull]
        private readonly RequestContext<bool> _requestSendTrustedPurchaseContext;
        [NotNull]
        private readonly RequestContext<string> _requestAuthorizeUserContext;
        [NotNull]
        private readonly RequestContext<SaveStateRequestResponse> _requestSaveStateContext;
        [NotNull]
        private readonly RequestContext<RestoreStateRequestResponse> _requestRestoreStateContext;

        [NotNull]
        public AuthorizationToken AuthorizationToken { get; }
        [NotNull]
        public IReactiveProperty<bool> IsGeoIpEnabled { get; } = new ReactiveProperty<bool>(true);
        public bool ShouldSendLastSyncProgressTime { get; set; } = true;

        public bool IsDisposed => _disposables.IsDisposed;
        public bool IsOffline => _interactionsProvider.URL.IsOffline;

        public ServerApi(
            EndpointUrl url,
            [NotNull] INetworkClient networkClient,
            [NotNull] GeneralPrefs generalPrefs,
            [NotNull] AppStatePrefs appStatePrefs,
            [NotNull] IAuthorizationConfigBuilder authConfig,
            [NotNull] PlatformAPI platform)
        {
            _disposableDependencies = new List<ICancelable>(5);
            _generalPrefs = generalPrefs.AddTo(_disposableDependencies);
            _appStatePrefs = appStatePrefs.AddTo(_disposableDependencies);
            _authConfigBuilder = authConfig.AddTo(_disposableDependencies);
            _platform = platform.AddTo(_disposableDependencies);
            AuthorizationToken = new AuthorizationToken(generalPrefs);
            _interactionsProvider = new ServerInteractionsProvider(url, networkClient, GetAuthorizationTokenAsync).AddTo(_disposables).AddTo(_disposableDependencies);

            _requestNewTokenContext = new RequestContext<string>
            (
                new SingleModeRequestTools<string>(),
                independentModeRequestTools: null,
                NewTokenResponseHandler
            ).AddTo(_disposables);
            _requestContextContext = new RequestContext<CampaignsContext>
            (
                new SingleModeRequestTools<CampaignsContext>(),
                independentModeRequestTools: null,
                GetContextResponseHandler
            ).AddTo(_disposables);
            _requestStoreEventsContext = new RequestContext<bool>
            (
                singleModeRequestTools: null,
                new IndependentModeRequestTools(),
                StoreEventsResponseHandler
            ).AddTo(_disposables);
            _requestVerifyPurchaseContext = new RequestContext<PurchaseVerificationResult>
            (
                singleModeRequestTools: null,
                new IndependentModeRequestTools(),
                VerifyPurchaseResponseHandler
            ).AddTo(_disposables);
            _requestSendTrustedPurchaseContext = new RequestContext<bool>
            (
                singleModeRequestTools: null,
                new IndependentModeRequestTools(),
                SendTrustedPurchaseResponseHandler
            ).AddTo(_disposables);
            _requestAuthorizeUserContext = new RequestContext<string>
            (
                new SingleModeRequestTools<string>(),
                independentModeRequestTools: null,
                AuthorizeUserResponseHandler
            ).AddTo(_disposables);
            _requestSaveStateContext = new RequestContext<SaveStateRequestResponse>
            (
                singleModeRequestTools: null,
                new IndependentModeRequestTools(),
                SaveStateResponseHandler
            ).AddTo(_disposables);
            _requestRestoreStateContext = new RequestContext<RestoreStateRequestResponse>
            (
                singleModeRequestTools: null,
                new IndependentModeRequestTools(),
                RestoreStateResponseHandler
            ).AddTo(_disposables);
        }

        void IInitializable.Initialize()
        {
            ThrowIfDisposed();
            var authConfig = _authConfigBuilder.AuthorizationConfig;
            if (authConfig.LanguageCode != _generalPrefs.LanguageCode.Value || authConfig.CountryCode != _generalPrefs.CountryCode.Value)
            {
                ApplyLocaleChanges(authConfig.LanguageCode, authConfig.CountryCode);
            }

            _generalPrefs.AttAuthorizationStatus
                .SkipLatestValueOnSubscribe()
                .Subscribe(_ => ResetAuthorizationToken())
                .AddTo(_disposables);
            IsGeoIpEnabled
                .SkipLatestValueOnSubscribe()
                .Subscribe(_ => ResetAuthorizationToken())
                .AddTo(_disposables);
            _generalPrefs.AdjustId
                .SkipLatestValueOnSubscribe()
                .Subscribe(_ => ResetAuthorizationToken())
                .AddTo(_disposables);
            _generalPrefs.IsGdprGranted
                .Subscribe(_ => RefreshGpsAdId())
                .AddTo(_disposables);
            _generalPrefs.SyncStateEnabled
                .Where(enabled => enabled)
                .Subscribe(_ =>
                {
                    ResetAuthorizationToken();
                    UniTask.RunOnThreadPool(() => this.GetAuthorizationTokenUntilSuccessAsync(_disposables.GetOrCreateToken())).Forget();
                })
                .AddTo(_disposables);
        }

        void IDisposable.Dispose()
        {
            _disposables.Release();
        }

        public void ChangeServerUrl(EndpointUrl url)
        {
            if (_interactionsProvider.URL == url)
            {
                return;
            }

            _interactionsProvider.ChangeServerUrl(url);
            ResetAuthorizationToken();
        }

        /// <returns>
        /// true if locale was changed
        /// </returns>
        public bool RefreshLocale([NotNull] LocaleData localeData)
        {
            _logger.Log($"Refresh locale with: \n{localeData}");
            ThrowIfDisposed();
            var authConfig = _authConfigBuilder.AuthorizationConfig;
            if (authConfig.LanguageCode != localeData.LanguageWithRegion || authConfig.CountryCode != localeData.CountryCode)
            {
                authConfig.LanguageCode = localeData.LanguageWithRegion;
                authConfig.CountryCode = localeData.CountryCode;
                _logger.Log($"Locale has been changed to: {localeData}");
                ApplyLocaleChanges(authConfig.LanguageCode, authConfig.CountryCode);
                return true;
            }
            return false;
        }

        private void ApplyLocaleChanges(string newLanguageCode, string newCountryCode)
        {
            ThrowIfDisposed();
            using (_generalPrefs.MultipleChangeScope())
            {
                _generalPrefs.LanguageCode.Value = newLanguageCode;
                _generalPrefs.CountryCode.Value = newCountryCode;
                _logger.Log($"Locale has been changed: LanguageCode={newLanguageCode}, countryCode={newCountryCode}");
                ResetAuthorizationToken();
            }
        }

        public void RefreshGpsAdId()
        {
            ThrowIfDisposed();
            using (_generalPrefs.MultipleChangeScope())
            {
                var newValue = _generalPrefs.IsGdprGranted.Value ? _platform.GetGpsAdId() : null;
                if (newValue != _generalPrefs.GpsAdId.Value)
                {
                    _generalPrefs.GpsAdId.Value = newValue;
                    ResetAuthorizationToken();
                }
            }
        }

        public UniTask CancelAllServerInteractions(bool includeProgress = false)
        {
            _logger.Log($"{nameof(CancelAllServerInteractions)} has been called");
            var context = CancelAllContextLoadings();
            var token = CancelAllTokenLoadings();
            CancelAllEventsSendings();
            CancelAllVerifyPurchaseRequests();
            if (includeProgress)
            {
                CancelAllAuthorizeUserRequests();
                CancelAllRestoreStateRequests();
                CancelAllSaveStateRequests();
            }
            ResetAuthorizationToken();
            _logger.Log($"{nameof(CancelAllServerInteractions)} returns promise...");
            return UniTask.WhenAll(context, token);
        }

        private void HandleBannedResponse()
        {
            _logger.Log("Player was banned, cancel trying to send requests");
            CancelAllTokenLoadings();
            CancelAllContextLoadings();
            CancelAllEventsSendings();
            CancelAllVerifyPurchaseRequests();
            CancelAllSendTrustedPurchaseRequests();
            CancelAllAuthorizeUserRequests();
            CancelAllSaveStateRequests();
            CancelAllRestoreStateRequests();
            AuthorizationToken.AuthToken = null;
        }

        private void ThrowIfDisposed()
        {
            if (IsDisposed)
            {
                throw new OperationCanceledException($"Cannot create request, because {typeof(ServerApi)} has been disposed.");
            }
            foreach (var dependency in _disposableDependencies)
            {
                if (dependency.IsDisposed)
                {
                    throw new OperationCanceledException($"Cannot create request, because {dependency.GetType()} has been disposed.");
                }
            }
        }
    }
}