using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine.Networking;

namespace Magify
{
    internal partial class ServerApi
    {
        [NotNull, ItemNotNull]
        private static readonly string[] _saveProgressExpectedHeaders = new string[]
        {
            "date",
        };

        public UniTask<SaveStateRequestResponse> SaveState([NotNull] string appState, [CanBeNull] int? weight, CancellationToken cancellationToken)
        {
            ThrowIfDisposed();
            var config = new RequestConfig(
                ApplicationStateApiMethods.SaveProgress(_platform.RuntimePlatform),
                createSaveStateRequestPayload,
                useWebRequestWithRepeats: false,
                isAuthTokenRequired: true);
            var request = _interactionsProvider.MakeAsyncRequest(
                in _requestSaveStateContext,
                in config,
                cancellationToken
            );
            return request.HandleByDefault(config, rejectedResult: SaveStateRequestResponse.Fail());

            WebRequestMessage createSaveStateRequestPayload()
            {
                ThrowIfDisposed();
                return new WebRequestMessage
                {
                    AuthToken = default,
                    Method = ApplicationStateApiMethods.SaveProgress(_platform.RuntimePlatform),
                    Payload = new SaveStatePayload
                    {
                        Progress = appState,
                        Weight = weight,
                        LastSyncTime = ShouldSendLastSyncProgressTime ? _appStatePrefs.LastSyncTime.Value : null,
                    },
                    ExpectedHeaders = _saveProgressExpectedHeaders,
                };
            }
        }

        private RepeatState SaveStateResponseHandler(WebResponseMessage response, out SaveStateRequestResponse result)
        {
            _logger.Log($"{nameof(SaveStateResponseHandler)} called");
            if (response.Result == UnityWebRequest.Result.Success)
            {
                _logger.Log("State successfully saved");
                result = SaveStateRequestResponse.Success();
                if (response.TryGetUnixTimeSeconds(out var unixTimeMs))
                {
                    _appStatePrefs.LastSyncTime.Value = unixTimeMs;
                }
                return RepeatState.Finish;
            }

            if (!string.IsNullOrEmpty(response.Text))
            {
                var errorResponse = ServerApiUtils.TryExtractErrorResponse(response.Text, _logger);
                if (ServerApiUtils.TryHandleDefaultErrorCodes(errorResponse?.Error, response.RequestMessage.Method, AuthorizationToken, _logger, HandleBannedResponse, out var repeatState))
                {
                    result = SaveStateRequestResponse.Fail();
                    return repeatState;
                }
                switch (errorResponse?.Error?.Code)
                {
                    case ErrorCode.SyncProgress_SaveConflict:
                        _logger.Log("A conflict error was received while requesting to save progress");
                        result = SaveStateRequestResponse.Conflict();
                        return RepeatState.Finish;

                    case ErrorCode.SyncProgress_ForceUpdateRequest:
                        _logger.Log("A force update progress error was received while requesting to save progress");
                        result = SaveStateRequestResponse.ForceUpdateRequest();
                        return RepeatState.Finish;
                }
            }

            result = SaveStateRequestResponse.Fail();
            return RepeatState.Finish;
        }

        public void CancelAllSaveStateRequests()
        {
            _logger.Log($"{nameof(CancelAllSaveStateRequests)} has been called");
            _requestSaveStateContext.CancelAllRequests();
        }
    }
}