using System.Collections.Generic;
using System.Threading;
using Cysharp.Threading.Tasks;
using Magify.Model;
using UnityEngine.Networking;
using JetBrains.Annotations;

namespace Magify
{
    internal partial class ServerApi
    {
        public UniTask<bool> SendEvents<T>(EventType eventType, [NotNull] IList<T> events, CancellationToken cancellationToken)
            where T : class, IAnalyticsEvent
        {
            ThrowIfDisposed();
            var config = new RequestConfig(
                WebRequestMethods.Store,
                createStoreEventsRequestPayload,
                useWebRequestWithRepeats: false,
                isAuthTokenRequired: true);
            var request = _interactionsProvider.MakeAsyncRequest(
                in _requestStoreEventsContext,
                in config,
                cancellationToken
            );
            return request.HandleByDefault(config, rejectedResult: false);

            WebRequestMessage createStoreEventsRequestPayload()
            {
                ThrowIfDisposed();
                return new WebRequestMessage
                {
                    Method = WebRequestMethods.Store,
                    Payload = new StorePayload
                    {
                        Model = eventType,
                        Objects = events,
                    },
                };
            }
        }

        private RepeatState StoreEventsResponseHandler(WebResponseMessage response, out bool result)
        {
            if (response.Result == UnityWebRequest.Result.Success)
            {
                result = true;
                return RepeatState.Finish;
            }

            if (!string.IsNullOrEmpty(response.Text))
            {
                var errorResponse = ServerApiUtils.TryExtractErrorResponse(response.Text, _logger);
                if (ServerApiUtils.TryHandleDefaultErrorCodes(errorResponse?.Error, response.RequestMessage.Method, AuthorizationToken, _logger, HandleBannedResponse, out var repeatState))
                {
                    result = false;
                    return repeatState;
                }
            }

            result = false;
            return RepeatState.Finish;
        }

        public void CancelAllEventsSendings()
        {
            _logger.Log($"{nameof(CancelAllEventsSendings)} has been called");
            _requestStoreEventsContext.CancelAllRequests();
        }
    }
}