using System;
using System.Text;
using System.Threading;
using Cysharp.Threading.Tasks;
using Magify.Model;
using JetBrains.Annotations;
using UnityEngine;
using UnityEngine.Networking;

namespace Magify
{
    internal class NetworkClient : INetworkClient
    {
        private const int TimeoutSec = 35;

        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Network);
        [NotNull]
        private static readonly UTF8Encoding _utf8Encoding = new();
        [NotNull, ItemNotNull]
        private static readonly IUnityWebRequestBuilder[] _builders =
        {
            new SyncUnityWebRequestBuilder(),
            new UnityWebRequestOnMainThreadBuilder(),
        };

        public async UniTask<WebResponseMessage> SendAsync([NotNull] string url, WebRequestMessage message, CancellationToken cancellationToken)
        {
            try
            {
                _logger.Log($"SendAsync with message => {message}");
                using var request = await CreateRequest(url, message, cancellationToken);
                await request.SendWebRequest()!.WithCancellation(cancellationToken);
                cancellationToken.ThrowIfCancellationRequested();
                _logger.Log($"Finished SendAsync with message => {message} with result: {request.result}");
                return request.ToWebRequestResponse(message);
            }
            catch (OperationCanceledException e)
            {
                _logger.Log($"Cancelled web request: {message.Method} with {e.Message}");
                throw;
            }
            catch (UnityException e)
            {
                _logger.LogError("Failed to send web request with exception." +
                                 "Don't worry, Magify will try to send it bit later again." +
                                 "But you should send info about this error to Magify support." +
                                 $"\n{e}");
                return new WebResponseMessage
                {
                    Result = UnityWebRequest.Result.ConnectionError,
                    ResponseCode = -1,
                    Error = null,
                    Text = null,
                    Headers = null,
                    RequestMessage = message,
                };
            }
        }

        [NotNull, ItemNotNull]
        private static async UniTask<UnityWebRequest> CreateRequest(
            [NotNull] string url,
            WebRequestMessage message,
            CancellationToken cancellationToken)
        {
            var payload = new JsonRPCPayload(message.Method, message.Payload);
            var payloadString = JsonFacade.SerializeObject(payload);
            var payloadBytes = _utf8Encoding.GetBytes(payloadString);
            using var cts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
            cts.CancelAfter(millisecondsDelay: TimeoutSec * 1000);
            cancellationToken = cts.Token;

            UnityWebRequest request = null;
            foreach (var builder in _builders)
            {
                try
                {
                    request = await builder.Post(
                        url,
                        payloadBytes,
                        cancellationToken,
                        TimeoutSec);
                    if (request != null)
                    {
                        break;
                    }
                }
                catch (Exception e)
                {
                    _logger.Log($"Failed to create web request with builder {builder.GetType()} with exception: {e}");
                }
            }

            cancellationToken.ThrowIfCancellationRequested();
            if (request == null)
            {
                throw new MagifyFailedToBuildWebRequestException(_builders);
            }

            request.SetRequestHeader("Content-Type", "application/json");
            if (message.AuthToken != null)
            {
                request.SetRequestHeader("Authorization", message.AuthToken);
            }
            request.SetRequestHeader("Client-Timezone", DateTimeHelper.GetClientTimeZone());

            return request;
        }
    }
}