﻿using System.Collections.Generic;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine.Networking;

namespace Magify
{
    internal partial class AghanimServerApi
    {
        [CanBeNull, ItemNotNull]
        public UniTask<IReadOnlyList<AghanimProductCounting>> GetProductsCount(CancellationToken cancellationToken)
        {
            ThrowIfDisposed();
            var config = new RequestConfig(
                AghanimWebRequestMethods.GetProductsCount,
                createGetProductCountRequestPayload,
                useWebRequestWithRepeats: true,
                isAuthTokenRequired: true);
            var request = _interactionsProvider.MakeAsyncRequest(
                in _getProductCountContext,
                in config,
                cancellationToken);
            return request.HandleByDefault(config, rejectedResult: null);

            WebRequestMessage createGetProductCountRequestPayload()
            {
                return new WebRequestMessage
                {
                    Method = AghanimWebRequestMethods.GetProductsCount,
                    Payload = null,
                };
            }
        }

        private RepeatState GetProductCountResponseHandler(WebResponseMessage response, [CanBeNull, ItemNotNull] out IReadOnlyList<AghanimProductCounting> result)
        {
            if (response is { Result: UnityWebRequest.Result.Success, Text: not null })
            {
                _logger.Log($"Aghanim successfully created order. Result text={response.Text}");
                if (ServerApiUtils.TryDeserializeFromBasicResponse(response.Text, _logger, out AghanimProductCounting[] array, out var exception))
                {
                    result = array;
                    return RepeatState.Finish;
                }
                else
                {
                    _logger.LogWarning($"Failed to parse Aghanim's order response: {exception.Message}");
                    result = default;
                    return RepeatState.Wait;
                }
            }

            if (!string.IsNullOrEmpty(response.Text))
            {
                var errorResponse = ServerApiUtils.TryExtractErrorResponse(response.Text, _logger);
                if (ServerApiUtils.TryHandleDefaultErrorCodes(errorResponse?.Error, response.RequestMessage.Method, _authorizationToken, _logger, HandleBannedResponse, out var repeatState))
                {
                    result = null;
                    return repeatState;
                }
            }

            result = null;
            return RepeatState.Finish;
        }
    }
}