﻿using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine.Networking;

namespace Magify
{
    internal partial class AghanimServerApi
    {
        public UniTask<AghanimOrderStatus?> GetOrderStatus([NotNull] string orderId, CancellationToken cancellationToken)
        {
            ThrowIfDisposed();
            var config = new RequestConfig(
                AghanimWebRequestMethods.GetProductStatus,
                createGetProductStatusRequestPayload,
                useWebRequestWithRepeats: false,
                isAuthTokenRequired: true);
            var request = _interactionsProvider.MakeAsyncRequest(
                in _getOrderStatusContext,
                in config,
                cancellationToken);
            return request.HandleByDefault(config, rejectedResult: null);

            WebRequestMessage createGetProductStatusRequestPayload()
            {
                return new WebRequestMessage
                {
                    Method = AghanimWebRequestMethods.GetProductStatus,
                    Payload = new GetAghanimOrderStatus(orderId),
                };
            }
        }

        private RepeatState GetProductStatusResponseHandler(WebResponseMessage response, out AghanimOrderStatus? result)
        {
            if (response is { Result: UnityWebRequest.Result.Success, Text: not null })
            {
                _logger.Log($"Aghanim successfully created order. Result text={response.Text}");
                if (ServerApiUtils.TryDeserializeFromBasicResponse(response.Text, _logger, out result, out var exception))
                {
                    if (result is null or AghanimOrderStatus.Pending)
                    {
                        _logger.Log($"Aghanim order status is {result}, trying to get order status again.");
                    }
                    return RepeatState.Finish;
                }
                else
                {
                    _logger.LogWarning($"Failed to parse Aghanim's order response: {exception.Message}");
                    result = default;
                    return RepeatState.Wait;
                }
            }

            if (!string.IsNullOrEmpty(response.Text))
            {
                var errorResponse = ServerApiUtils.TryExtractErrorResponse(response.Text, _logger);
                if (ServerApiUtils.TryHandleDefaultErrorCodes(errorResponse?.Error, response.RequestMessage.Method, _authorizationToken, _logger, HandleBannedResponse, out var repeatState))
                {
                    result = null;
                    return repeatState;
                }
            }

            result = null;
            return RepeatState.Finish;
        }
    }
}