﻿using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine.Networking;

namespace Magify
{
    internal partial class AghanimServerApi
    {
        [CanBeNull, ItemCanBeNull]
        public UniTask<AghanimOrder> CreateOrder([NotNull] string productId, bool isSandbox, CancellationToken cancellationToken)
        {
            ThrowIfDisposed();
            var config = new RequestConfig(
                AghanimWebRequestMethods.CreateOrder,
                createCreateOrderRequestPayload,
                useWebRequestWithRepeats: true,
                isAuthTokenRequired: true);
            var request = _interactionsProvider.MakeAsyncRequest(
                in _createOrderContext,
                in config,
                cancellationToken);
            return request.HandleByDefault(config, rejectedResult: null);

            WebRequestMessage createCreateOrderRequestPayload()
            {
                return new WebRequestMessage
                {
                    Method = AghanimWebRequestMethods.CreateOrder,
                    Payload = new CreateAghanimOrderPayload(productId, isSandbox),
                };
            }
        }

        private RepeatState CreateOrderResponseHandler(WebResponseMessage response, [CanBeNull] out AghanimOrder result)
        {
            if (response is { Result: UnityWebRequest.Result.Success, Text: not null })
            {
                _logger.Log($"Aghanim successfully created order. Result text={response.Text}");
                if (ServerApiUtils.TryDeserializeFromBasicResponse(response.Text, _logger, out result, out var exception))
                {
                    return RepeatState.Finish;
                }
                else
                {
                    _logger.LogWarning($"Failed to parse Aghanim's order response: {exception.Message}");
                    result = default;
                    return RepeatState.Wait;
                }
            }

            if (!string.IsNullOrEmpty(response.Text))
            {
                var errorResponse = ServerApiUtils.TryExtractErrorResponse(response.Text, _logger);
                if (ServerApiUtils.TryHandleDefaultErrorCodes(errorResponse?.Error, response.RequestMessage.Method, _authorizationToken, _logger, HandleBannedResponse, out var repeatState))
                {
                    result = null;
                    return repeatState;
                }
            }

            result = null;
            return RepeatState.Finish;
        }
    }
}