﻿using System;
using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using Magify.Types;

namespace Magify
{
    internal class MinimalAnalyticsTracker : IAnalyticsTracker, IInitializable, IDisposable
    {
        private readonly IServerApi _serverApi;
        private readonly GeneralPrefs _generalPrefs;
        private readonly AppStatePrefs _appStatePrefs;
        [NotNull]
        private readonly AppVersionProvider _appVersionProvider;
        private readonly IDictionary<EventType, IAnalyticsEventHandler> _eventHandlers = new Dictionary<EventType, IAnalyticsEventHandler>();

        private readonly Subject<EventType> _onEventSent = new();
        private readonly Subject<(EventType, IAnalyticsEvent)> _onEventSentDetailed = new();
        private PlatformAPI _platform;

        public IObservable<EventType> OnEventSent => _onEventSent;
        public IObservable<(EventType Type, IAnalyticsEvent Event)> OnEventSentDetailed => _onEventSentDetailed;

        public MinimalAnalyticsTracker(
            IServerApi serverApi,
            GeneralPrefs generalPrefs,
            AppStatePrefs appStatePrefs,
            PlatformAPI platform,
            string storagePath,
            [NotNull] AppVersionProvider appVersionProvider)
        {
            _serverApi = serverApi;
            _generalPrefs = generalPrefs;
            _appStatePrefs = appStatePrefs;
            _platform = platform;
            _appVersionProvider = appVersionProvider;
            CreateEventHandlers(storagePath);
        }

        void IInitializable.Initialize()
        {
            foreach (var (_, value) in _eventHandlers)
            {
                value.Initialize();
            }
        }

        void IDisposable.Dispose()
        {
            _onEventSent?.Dispose();
            _onEventSentDetailed?.Dispose();
            _eventHandlers.Values.OfType<IDisposable>().ForEach(d => d.Dispose());
        }

        public void HandeWentForeground()
        {
            foreach (var (_, value) in _eventHandlers)
            {
                value.OnForeground();
            }
        }

        public void HandeWentBackground()
        {
            foreach (var (_, value) in _eventHandlers)
            {
                value.OnBackground();
            }
        }

        public void SetupAnalyticsConfig(AnalyticsConfiguration config)
        {
            foreach (var (_, value) in _eventHandlers)
            {
                value.SetupAnalyticsConfig(config);
            }
        }

        public void ResetAnalyticsConfig()
        {
            foreach (var (_, value) in _eventHandlers)
            {
                value.ResetAnalyticsConfig();
            }
        }

        public void TrackAppLaunch()
        {
            var @event = new AppLaunchEvent
            {
                SubscriptionStatus = _generalPrefs.SubscriptionStatus.Value,
                InAppStatus = _generalPrefs.InAppStatus.Value,
                AppVersion = _appVersionProvider.AppVersion,
                IsBasicIntegration = true,
            };
            HandleEvent(EventType.AppLaunch, @event);
        }

        public void TrackAppBackgroundEvent(long openTime, long closeTime, string appVersion)
        {
            var @event = new AppBackgroundEvent
            {
                SessionId = _generalPrefs.SessionId.Value,
                OpenTimestamp = openTime.TimestampFromLongMills(),
                CloseTimestamp = closeTime.TimestampFromLongMills(),
                AppVersion = appVersion,
                IsBasicIntegration = true,
            };
            HandleEvent(EventType.AppBackgroundEvent, @event);
        }

        public void TrackUsedApplicationDefaultFeaturesEvent([NotNull] string featureName)
        {
            var @event = new UsedApplicationDefaultFeaturesEvent
            {
                FeatureName = featureName,
            };
            HandleEvent(EventType.UsedApplicationDefaultFeatures, @event);
        }

        private void HandleEvent<T>(EventType eventType, T @event)
            where T : class, IAnalyticsEvent
        {
            ((IAnalyticsEventHandler<T>)_eventHandlers[eventType]).HandleEvent(@event);
            _onEventSent.OnNext(eventType);
            _onEventSentDetailed.OnNext((eventType, @event));
        }

        public void TrackApplovinAdsImpression(ApplovinAdsImpression data)
        {
            var @event = new AdsImpressionEvent(
                CampaignImpression.CreateBase(_generalPrefs.GlobalSessionCounter.Value),
                data,
                _generalPrefs.SubscriptionStatus.Value,
                _generalPrefs.InAppStatus.Value,
                LevelStateInfo.Empty,
                _generalPrefs.SessionId.Value,
                _appVersionProvider.AppVersion,
                isBasicIntegration: true);
            HandleEvent(EventType.ApplovinAdsImpression, @event);
        }

        public void TrackIronSourceAdsImpression(IronSourceAdsImpression data)
        {
            var @event = new AdsImpressionEvent(
                CampaignImpression.CreateBase(_generalPrefs.GlobalSessionCounter.Value),
                data,
                _generalPrefs.SubscriptionStatus.Value,
                _generalPrefs.InAppStatus.Value,
                LevelStateInfo.Empty,
                _generalPrefs.SessionId.Value,
                _appVersionProvider.AppVersion,
                isBasicIntegration: true);
            HandleEvent(EventType.IronSourceAdsImpression, @event);
        }

        public void TrackInAppPurchaseEvent(EventType type, PurchaseInfo purchase)
        {
            TrackPurchaseEvent(type, CampaignImpression.BuildExternalInAppImpression(purchase.Product.ProductId, _generalPrefs.GlobalSessionCounter.Value), purchase);
        }

        public void TrackSubscriptionPurchaseEvent(EventType type, PurchaseInfo purchase)
        {
            TrackPurchaseEvent(type, CampaignImpression.BuildExternalSubscriptionImpression(purchase.Product.ProductId, _generalPrefs.GlobalSessionCounter.Value), purchase);
        }

        private void TrackPurchaseEvent(EventType type, CampaignImpression impression, PurchaseInfo purchase)
        {
            var analyticsEvent = new ProductPurchaseEvent(
                impression,
                purchase,
                _generalPrefs.SubscriptionStatus.Value,
                _generalPrefs.InAppStatus.Value,
                LevelStateInfo.Empty,
                _generalPrefs.SessionId.Value,
                purchase.CustomStoreFront is { IsSome: true } ? purchase.CustomStoreFront.Value : _platform.GetStoreCountry(),
                _appVersionProvider.AppVersion,
                isBasicIntegration: true);
            HandleEvent(type, analyticsEvent);
        }

        private void CreateEventHandlers(string storagePath)
        {
            add<AppLaunchEvent>(EventType.AppLaunch);
            add<AppBackgroundEvent>(EventType.AppBackgroundEvent);

            add<AdsImpressionEvent>(EventType.ApplovinAdsImpression);
            add<AdsImpressionEvent>(EventType.IronSourceAdsImpression);

            add<ProductPurchaseEvent>(EventType.InApp);
            add<ProductPurchaseEvent>(EventType.TrialActivation);
            add<ProductPurchaseEvent>(EventType.PaidSubscriptionActivation);
            add<UsedApplicationDefaultFeaturesEvent>(EventType.UsedApplicationDefaultFeatures);
            return;

            void add<T>(EventType eventType) where T: class, IAnalyticsEvent
            {
                _eventHandlers.Add(eventType, new EventHandler<T>(eventType, _serverApi, _appStatePrefs, storagePath));
            }
        }
    }
}