using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    internal class OffersCollection : ICancelable, IEnumerable<LtoModel>
    {
        [NotNull, ItemNotNull]
        private readonly List<LtoModel> _offers = new();
        [NotNull]
        private readonly Dictionary<string, CancellationTokenSource> _cancellationTokenSources = new();

        public bool IsDisposed { get; private set; }

        [NotNull, ItemNotNull]
        public IReadOnlyCollection<LtoModel> Offers => _offers;

        public bool Contains(string offerName)
        {
            return Offers.Any(c => c.CampaignName == offerName);
        }

        [CanBeNull]
        public LtoModel GetModelBySpot(string spot)
        {
            return Offers.FirstOrDefault(c => c.Spot == spot);
        }

        public void Add(LtoModel offer)
        {
            _offers.Add(offer);
        }

        public void Remove([NotNull] LtoModel offer)
        {
            CancelCancellationTokenSource(offer.CampaignName);
            _offers.Remove(offer);
        }

        public void AttachCancellationTokenSource([NotNull] string offerName, CancellationTokenSource cts)
        {
            CancelCancellationTokenSource(offerName);
            _cancellationTokenSources[offerName] = cts;
        }

        private void CancelCancellationTokenSource([NotNull] string offerName)
        {
            if (_cancellationTokenSources.TryGetValue(offerName, out var cts))
            {
                cts?.Cancel();
                cts?.Dispose();
                _cancellationTokenSources.Remove(offerName);
            }
        }

        public void Dispose()
        {
            if (IsDisposed)
            {
                return;
            }
            foreach (var cts in _cancellationTokenSources.Values)
            {
                cts?.Cancel();
                cts?.Dispose();
            }
            _offers.Clear();
            _cancellationTokenSources.Clear();
            IsDisposed = true;
        }

        public IEnumerator<LtoModel> GetEnumerator()
        {
            return _offers.GetEnumerator();
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }
    }
}