﻿using System.Collections.Generic;
using JetBrains.Annotations;
using Magify.Model;

namespace Magify
{
    internal abstract class CreativeParser
    {
        public virtual bool IsValid(CreativeModel model)
        {
            return true;
        }

        public abstract ICreative Parse([NotNull] CreativeModel model, [CanBeNull] IReadOnlyDictionary<string, object> attributes);
    }

    internal abstract class CreativeParser<T> : CreativeParser
        where T : ICreative
    {
        public sealed override ICreative Parse(CreativeModel model, [CanBeNull] IReadOnlyDictionary<string, object> attributes)
        {
            return ParseCreative(model, attributes);
        }

        protected abstract T ParseCreative([NotNull] CreativeModel model, [CanBeNull] IReadOnlyDictionary<string, object> attributes);
    }

    internal class BundleCreativeParser : CreativeParser<BundleCreative>
    {
        public override bool IsValid(CreativeModel model)
        {
            return base.IsValid(model) && !string.IsNullOrWhiteSpace(model.Info?.PageUrl);
        }

        protected override BundleCreative ParseCreative(CreativeModel model, IReadOnlyDictionary<string, object> attributes)
        {
            return new BundleCreative
            {
                Attributes = attributes,
                Url = model.Info!.PageUrl,
                ButtonCloseTimeout = (float)(model.Info.ShowButtonTimeout?.TotalSeconds ?? 0),
                Background = model.Background.AsCreativeBackground()
            };
        }
    }

    internal class CustomCreativeParser : CreativeParser<CustomCreative>
    {
        public override bool IsValid(CreativeModel model)
        {
            return base.IsValid(model) && !string.IsNullOrWhiteSpace(model.Info?.Id);
        }

        protected override CustomCreative ParseCreative(CreativeModel model, IReadOnlyDictionary<string, object> attributes)
        {
            return new CustomCreative
            {
                Attributes = attributes,
                ScreenId = model.Info!.Id,
            };
        }
    }

    internal class ImageCreativeParser : CreativeParser<ImageCreative>
    {
        public override bool IsValid(CreativeModel model)
        {
            return base.IsValid(model) && model.Info != null && (!string.IsNullOrWhiteSpace(model.Info.Portrait) || !string.IsNullOrWhiteSpace(model.Info.Landscape));
        }

        protected override ImageCreative ParseCreative(CreativeModel model, IReadOnlyDictionary<string, object> attributes)
        {
            return new ImageCreative
            {
                Attributes = attributes,
                Portrait = model.Info!.Portrait,
                Landscape = model.Info!.Landscape,
                ButtonCloseTimeout = (float)(model.Info.ShowButtonTimeout?.TotalSeconds ?? 0),
                Background = model.Background.AsCreativeBackground()
            };
        }
    }

    internal class NotificationAlertCreativeParser : CreativeParser<NotificationAlertCreative>
    {
        public override bool IsValid(CreativeModel model)
        {
            return base.IsValid(model) && !string.IsNullOrWhiteSpace(model.Info?.Title);
        }

        protected override NotificationAlertCreative ParseCreative(CreativeModel model, IReadOnlyDictionary<string, object> attributes)
        {
            return new NotificationAlertCreative
            {
                Attributes = attributes,
                Title = model.Info!.Title,
                Message = model.Info.Message,
                CloseButton = model.Info.CloseBtn,
            };
        }
    }

    internal class NoUiCreativeParser : CreativeParser<NoUiCreative>
    {
        protected override NoUiCreative ParseCreative(CreativeModel model, IReadOnlyDictionary<string, object> attributes)
        {
            return new NoUiCreative { Attributes = attributes };
        }
    }

    internal class PlayableCreativeParser : CreativeParser<PlayableCreative>
    {
        public override bool IsValid(CreativeModel model)
        {
            return base.IsValid(model) && !string.IsNullOrWhiteSpace(model.Info?.PageUrl);
        }

        protected override PlayableCreative ParseCreative(CreativeModel model, IReadOnlyDictionary<string, object> attributes)
        {
            return new PlayableCreative
            {
                Attributes = attributes,
                Url = model.Info!.PageUrl,
                ButtonCloseTimeout = (float)(model.Info.ShowButtonTimeout?.TotalSeconds ?? 0),
                Background = model.Background.AsCreativeBackground()
            };
        }
    }

    internal class SystemAlertCreativeParser : CreativeParser<SystemAlertCreative>
    {
        public override bool IsValid(CreativeModel model)
        {
            return base.IsValid(model) && !string.IsNullOrWhiteSpace(model.Info?.Title);
        }

        protected override SystemAlertCreative ParseCreative(CreativeModel model, IReadOnlyDictionary<string, object> attributes)
        {
            return new SystemAlertCreative
            {
                Attributes = attributes,
                Title = model.Info!.Title,
                Message = model.Info.Message,
                CloseButton = model.Info.CloseBtn,
                ActionButton = model.Info.ActionBtn,
            };
        }
    }
}