using System;
using System.Collections.Generic;
using System.Linq;
using Magify.Model;
using JetBrains.Annotations;

namespace Magify
{
    internal class LimitsHolder : IContextListener
    {
        [NotNull]
        private readonly GeneralPrefs _generalPrefs;
        [NotNull]
        private readonly SessionCounter _sessionCounter;

        [CanBeNull]
        private Magify.Model.Limits _limits;
        [CanBeNull]
        internal Magify.Model.Limits Limits => _limits;
        [NotNull]
        private readonly object _lock = new();

        public ConfigScope SuitableScope => ConfigScope.Limits;

        public LimitsHolder([NotNull] GeneralPrefs generalPrefs, [NotNull] SessionCounter sessionCounter)
        {
            _generalPrefs = generalPrefs;
            _sessionCounter = sessionCounter;
        }

        void IContextListener.UpdateContext([NotNull] CampaignsContext context, ContextKind kind)
        {
            lock (_lock)
            {
                switch (kind)
                {
                    case ContextKind.Default when _limits == null:
                    case ContextKind.Saved or ContextKind.Downloaded:
                        _limits = context.Limits;
                        _sessionCounter.Interval.Value = _limits?.SessionsInterval ?? SessionCounter.DefaultSessionInterval;
                        break;
                }
            }
        }

        public void Reset()
        {
            lock (_lock)
            {
                _limits = null;
            }
        }

        public (IEnumerable<TemporaryLimitsGroup> TemporaryLimits, object Lock) IterateTemporaryLimitsForProductWithLock(string productId)
        {
            return (_limits?.TempLimits?.Count switch
            {
                0 or null => ArraySegment<TemporaryLimitsGroup>.Empty,
                _ => _limits.TempLimits.Where(limit => limit.IncludeIfAnyPurchased?.Any(p => p.Product == productId) is true),
            }, _lock);
        }

        public LimitsModel GetActiveLimits()
        {
            lock (_lock)
            {
                var tempLimits = GetActiveTemporaryLimitsThreadUnsafe();
                var tempGlobalInterval = tempLimits.Max(p => p.GlobalInterval);
                var tempInterInterval = tempLimits.Max(p => p.InterstitialInterval);
                var tempRewardInterInterval = tempLimits.Max(p => p.RewardedInterstitialInterval);

                var campaignTypeLimits = GetCampaignTypeLimitsThreadUnsafe(tempLimits);
                return new LimitsModel
                {
                    GlobalInterval = TimeExtensions.MaxOf(_limits?.GlobalInterval, tempGlobalInterval),
                    SessionsInterval = _limits?.SessionsInterval,
                    InterstitialInterval = TimeExtensions.MaxOf(_limits?.InterstitialInterval, tempInterInterval),
                    RewardInterstitialInterval = TimeExtensions.MaxOf(_limits?.RewardedInterstitialInterval, tempRewardInterInterval),
                    ImpressionsPerSession = _limits?.ImpressionsPerSession,
                    InterstitialsPerSession = _limits?.InterstitialLimits?.SessionLimit,
                    CampaignTypeLimits = campaignTypeLimits,
                };
            }
        }

        internal Dictionary<CampaignType, ImpressionLimitsModel> GetCampaignTypeLimits(IReadOnlyCollection<TemporaryLimitsGroup> temporaryLimits)
        {
            lock (_lock)
            {
                return GetCampaignTypeLimitsThreadUnsafe(temporaryLimits);
            }
        }

        private Dictionary<CampaignType, ImpressionLimitsModel> GetCampaignTypeLimitsThreadUnsafe(IReadOnlyCollection<TemporaryLimitsGroup> temporaryLimits)
        {
            return EnumExtensions.GetValues<CampaignType>().Distinct().ToDictionary(
                campaignType => campaignType,
                campaignType =>
                {
                    var typeLimit = _limits?.GetCampaignTypeImpressionLimits(campaignType);
                    var periodTypeLimits = temporaryLimits.Select(limitGroup =>
                        {
                            var count = limitGroup.GetCampaignTypePeriodLimits(campaignType)?.Count;
                            return count == null
                                ? null
                                : new TemporaryImpressionLimit
                                {
                                    LimitId = limitGroup.GroupId,
                                    ImpressionsPerPeriod = count.Value
                                };
                        })
                        .Where(y => y != null)
                        .ToList();

                    return new ImpressionLimitsModel
                    {
                        GlobalLimit = typeLimit?.GlobalLimit,
                        SessionLimit = typeLimit?.SessionLimit,
                        DailyLimit = typeLimit?.DailyLimit,
                        PeriodLimit = typeLimit?.PeriodLimit,
                        Period = typeLimit?.Period,
                        TempLimits = periodTypeLimits
                    };
                });
        }

        // ToDo: check if IEnumerable is -better by performance?
        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        private IReadOnlyCollection<TemporaryLimitsGroup> GetActiveTemporaryLimitsThreadUnsafe()
        {
            if (_limits?.TempLimits?.Count is null or 0)
            {
                return ArraySegment<TemporaryLimitsGroup>.Empty;
            }

            var productsTime = _generalPrefs.ProductsReceiptTimestamps;
            var activeLimits = new List<TemporaryLimitsGroup>();
            foreach (var limit in _limits.TempLimits.Where(c => c.IncludeIfAnyPurchased != null))
            {
                foreach (var productItem in limit.IncludeIfAnyPurchased!)
                {
                    var hasProduct = productsTime.TryGetValue(productItem.Product, out var productReceiptTime);
                    if (hasProduct && productReceiptTime + limit.PeriodOfLimits > DateTime.UtcNow)
                    {
                        activeLimits.Add(limit);
                        break;
                    }
                }
            }
            return activeLimits;
        }
    }

    internal static class LimitsExtensions
    {
        [CanBeNull]
        internal static ImpressionLimits GetCampaignTypeImpressionLimits(this Model.Limits limits, CampaignType type)
        {
            return type switch
            {
                CampaignType.Interstitial => limits.InterstitialLimits,
                CampaignType.InApp => limits.InAppLimits,
                CampaignType.LtoInApp => limits.LtoInAppLimits,
                CampaignType.Subscription => limits.SubscriptionLimits,
                CampaignType.LtoSubscription => limits.LtoSubscriptionLimits,
                CampaignType.CrossPromo => limits.CrossPromoLimits,
                CampaignType.LtoCrossPromo => limits.LtoCrossPromoLimits,
                CampaignType.Notification => limits.NotificationLimits,
                CampaignType.External => limits.ExternalLimits,
                CampaignType.LtoExternal => limits.LtoExternalLimits,
                CampaignType.RateReview => limits.RateReviewLimits,
                _ => null
            };
        }

        [CanBeNull]
        internal static ImpressionPeriodLimit GetCampaignTypePeriodLimits(this TemporaryLimitsGroup limit, CampaignType type)
        {
            return type switch
            {
                CampaignType.Interstitial => limit.InterstitialLimits,
                CampaignType.InApp => limit.InAppLimits,
                CampaignType.LtoInApp => limit.LtoInAppLimits,
                CampaignType.Subscription => limit.SubscriptionLimits,
                CampaignType.LtoSubscription => limit.LtoSubscriptionLimits,
                CampaignType.CrossPromo => limit.CrossPromoLimits,
                CampaignType.LtoCrossPromo => limit.LtoCrossPromoLimits,
                CampaignType.Notification => limit.NotificationLimits,
                CampaignType.External => limit.ExternalLimits,
                CampaignType.LtoExternal => limit.LtoExternalLimits,
                CampaignType.RateReview => limit.RateReviewLimits,
                CampaignType.Bonus => limit.BonusLimits,
                CampaignType.LtoBonus => limit.LtoBonusLimits,
                _ => null
            };
        }
    }
}