﻿using System;
using System.Collections.Generic;
using System.IO;
using JetBrains.Annotations;
using UnityEngine;

namespace Magify
{
    internal class CountersStorage : IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Storage);

        [NotNull]
        private readonly string _folder;
        [NotNull]
        private readonly Dictionary<CounterType, BinaryStorage> _storages = new();
        [NotNull]
        private readonly object _lock = new();

        public CountersStorage([NotNull] string folder)
        {
            _folder = folder;
        }

        [NotNull]
        public BinaryStorage this[CounterType type]
        {
            get
            {
                BinaryStorage storage;
                lock (_lock)
                {
                    if (_storages.TryGetValue(type, out storage) && storage != null)
                        return storage;
                }
                storage = BinaryStorage
                    .Construct(Path.Combine(_folder, $"{type.ToEnumString()}.bin"))
                    .AddEntry(BinaryTypeString.Shared)
                    .AddEntry(BinaryTypeInt32.Shared)
                    .SupportDictionariesOf<string, int>()
                    .Build();
                lock (_lock)
                {
                    return _storages[type] = storage;
                }
            }
        }

        public void Dispose()
        {
            lock (_lock)
            {
                foreach (var pair in _storages)
                {
                    pair.Value?.Dispose();
                }
                _storages.Clear();
            }
        }

        /// <returns>Content of all counters as dictionary in JSON format.</returns>
        [NotNull]
        internal string GetFileContent()
        {
            Dictionary<CounterType, string> map;
            lock (_lock)
            {
                map = new Dictionary<CounterType, string>(_storages.Count);
                foreach (var (counter, storage) in _storages)
                {
                    var content = storage!.GetFileContent();
                    map.Add(counter, content);
                }
            }
            return JsonFacade.SerializeObject(map); // ToDo: optimize serialization.
        }

        /// <summary>
        /// Rewrite content of all counters from JSON map.
        /// </summary>
        internal void RewriteContent([NotNull] string countersContentMap)
        {
            var map = JsonFacade.DeserializeObject<Dictionary<CounterType, string>>(countersContentMap); // ToDo: optimize serialization.
            if (map == null || map.Count == 0)
            {
                return;
            }
            foreach (var (counter, storageContent) in map)
            {
                _logger.Log($"Rewriting content of counter storage: {counter}, content: \n{storageContent}");
                this[counter].RewriteContent(storageContent ?? string.Empty);
            }
        }
    }
}