using System;
using System.Text;

namespace Magify
{
    internal readonly struct CounterKey
    {
        [Flags]
        public enum Flags
        {
            None = 0,
            CampaignNameProp = 1 << 0,
            CampaignTypeProp = 1 << 1,
            NestedNameProp = 1 << 2,
            NestedTypeProp = 1 << 3,
            SourceProp = 1 << 4,
            SourceTypeProp = 1 << 5,
            LimitIdProp = 1 << 6
        }

        private static class SerializationKeys
        {
            public const string CampaignName = "c";
            public const string CampaignType = "ct";
            public const string NestedName = "n";
            public const string NestedType = "nt";
            public const string Source = "s";
            public const string SourceType = "st";
            public const string LimitId = "l";
        }

        private static readonly StringBuilder _keyBuilder = new();

        public const char KeySplitter = 'Ў';

        public string CampaignName { get; init; }
        public string NestedName { get; init; }
        public string NestedType { get; init; }
        public CampaignType? CampaignType { get; init; }
        public string Source { get; init; }
        public SourceType? SourceType { get; init; }
        public long? LimitId { get; init; }

        public Flags GetUsedProperties()
        {
            var usedFlags = Flags.None;
            usedFlags |= !string.IsNullOrEmpty(CampaignName) ? Flags.CampaignNameProp : Flags.None;
            usedFlags |= CampaignType.HasValue ? Flags.CampaignTypeProp : Flags.None;
            usedFlags |= !string.IsNullOrEmpty(NestedName) ? Flags.NestedNameProp : Flags.None;
            usedFlags |= !string.IsNullOrEmpty(NestedType) ? Flags.NestedTypeProp : Flags.None;
            usedFlags |= !string.IsNullOrEmpty(Source) ? Flags.SourceProp : Flags.None;
            usedFlags |= SourceType.HasValue ? Flags.SourceTypeProp : Flags.None;
            usedFlags |= LimitId.HasValue ? Flags.LimitIdProp : Flags.None;
            return usedFlags;
        }

        #region Parsing

        public override string ToString()
        {
            _keyBuilder.Clear();

            add(SerializationKeys.CampaignName, CampaignName);
            add(SerializationKeys.CampaignType, CampaignType?.ToEnumString());
            add(SerializationKeys.NestedName, NestedName);
            add(SerializationKeys.NestedType, NestedType);
            add(SerializationKeys.Source, Source);
            add(SerializationKeys.SourceType, SourceType?.ToEnumString());
            add(SerializationKeys.LimitId, LimitId?.ToString());

            return _keyBuilder.ToString();

            void add(string key, string value)
            {
                if (string.IsNullOrEmpty(value))
                {
                    return;
                }

                if (_keyBuilder.Length > 0)
                {
                    _keyBuilder.Append(KeySplitter);
                }

                _keyBuilder.Append(key);
                _keyBuilder.Append('=');
                _keyBuilder.Append(value);
            }
        }

        public static CounterKey FromString(string sourceString)
        {
            string campaignName = null;
            CampaignType? campaignType = null;
            string nestedName = null;
            string nestedType = null;
            string source = null;
            SourceType? sourceType = null;
            long? limitId = null;

            sourceString.Split(KeySplitter, StringSplitOptions.RemoveEmptyEntries, (ref ReadOnlySpan<char> entry) =>
            {
                var equalsIndex = entry.IndexOf('=');
                if (equalsIndex == -1)
                {
                    return;
                }
                var key = entry[..equalsIndex];
                var value = entry[(equalsIndex + 1)..];

                if (key.Equals(SerializationKeys.CampaignName, StringComparison.Ordinal))
                {
                    campaignName = value.ToString();
                }
                else if (key.Equals(SerializationKeys.CampaignType, StringComparison.Ordinal))
                {
                    campaignType = value.ToEnumMember<CampaignType>();
                }
                else if (key.Equals(SerializationKeys.NestedName, StringComparison.Ordinal))
                {
                    nestedName = value.ToString();
                }
                else if (key.Equals(SerializationKeys.NestedType, StringComparison.Ordinal))
                {
                    nestedType = value.ToString();
                }
                else if (key.Equals(SerializationKeys.Source, StringComparison.Ordinal))
                {
                    source = value.ToString();
                }
                else if (key.Equals(SerializationKeys.SourceType, StringComparison.Ordinal))
                {
                    sourceType = value.ToEnumMember<SourceType>();
                }
                else if (key.Equals(SerializationKeys.LimitId, StringComparison.Ordinal))
                {
                    limitId = long.Parse(value);
                }
            });

            return GetKey(
                campaignName: campaignName,
                campaignType: campaignType,
                nestedName: nestedName,
                nestedType: nestedType,
                source: source,
                sourceType: sourceType,
                limitId: limitId);
        }

        #endregion

        #region Helpers

        public static CounterKey GetKey(
            string campaignName = null,
            CampaignType? campaignType = null,
            string nestedName = null,
            string nestedType = null,
            string source = null,
            SourceType? sourceType = null,
            long? limitId = null)
        {
            return new CounterKey
            {
                CampaignName = campaignName,
                CampaignType = campaignType,
                NestedName = nestedName,
                NestedType = nestedType,
                Source = source,
                SourceType = sourceType,
                LimitId = limitId
            };
        }

        public static CounterKey GetKeyCampaignBySourceType(string campaignName, SourceType type)
        {
            return GetKey(campaignName: campaignName, sourceType: type);
        }

        public static CounterKey GetKeyCampaignBySource(string campaignName, string sourceName)
        {
            return GetKey(campaignName: campaignName, source: sourceName);
        }

        public static CounterKey GetKeyCampaignBySource(string campaignName, string sourceName, SourceType type)
        {
            return GetKey(campaignName: campaignName, source: sourceName, sourceType: type);
        }

        public static CounterKey GetKeyCampaignTypeBySourceType(CampaignType campaignType, SourceType sourceType)
        {
            return GetKey(campaignType: campaignType, sourceType: sourceType);
        }

        public static CounterKey GetKeyCampaignTypeByPeriodLimit(CampaignType campaignType, long limitId)
        {
            return GetKey(campaignType: campaignType, limitId: limitId);
        }

        #endregion
    }
}