using System;
using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using UnityEngine.Pool;
using static Magify.CounterScopeExt;

namespace Magify
{
    internal interface ICounter
    {
        public bool IsSupportScope(CounterScope scope);
        public void ResetAll(CounterScope scope);
        public void ResetAll(Func<CounterScope, CounterKey, bool> predicate);
    }

    internal class Counter<T> : ICounter
        where T : unmanaged, Enum, IComparable
    {
        private readonly BinaryStorage _storage;
        private readonly T[] _scopes;
        private readonly IDictionary<string, int>[] _scopesCounters;
        private readonly bool[] _supportedBaseScopes;

        [CanBeNull]
        private readonly Func<CounterScope, CounterKey, int?> _alternativeCountersStorage;

        public Counter(BinaryStorage storage, [CanBeNull] Func<CounterScope, CounterKey, int?> alternativeCountersStorage = null)
        {
            _storage = storage;
            _alternativeCountersStorage = alternativeCountersStorage;
            _scopes = EnumExtensions.GetValues<T>(out var length).ToArray(length);
            _scopesCounters = new IDictionary<string, int>[Scopes.Length];
            _supportedBaseScopes = new bool[Scopes.Length];
            foreach (var scope in _scopes)
            {
                var index = ToNumber(scope);
                var key = ToBase(scope).ToEnumString();
                _scopesCounters[index] = storage.GetDictionary<string, int>(key);
                _supportedBaseScopes[index] = true;
            }
        }

        public int this[T scope, CounterKey key]
        {
            get
            {
                var baseScope = ToBase(scope);
                var scopeMap = _scopesCounters[ToNumber(scope)];
                var keyStr = key.ToString();
                if (baseScope == CounterScope.Global && _alternativeCountersStorage != null && !scopeMap.ContainsKey(keyStr))
                {
                    var migratedCounter = _alternativeCountersStorage(baseScope, key);
                    if (migratedCounter != null)
                    {
                        var value = Math.Max(migratedCounter.Value, 0);
                        scopeMap[keyStr] = value;
                        return value;
                    }
                }
                return scopeMap.GetValueOrDefault(keyStr);
            }
            set => _scopesCounters[ToNumber(scope)][key.ToString()] = Math.Max(value, 0);
        }

        public IDisposable MultipleChangeScope()
        {
            return _storage.MultipleChangeScope();
        }

        public void Increment(T scope, CounterKey key)
        {
            this[scope, key]++;
        }

        public void Decrement(T scope, CounterKey key)
        {
            this[scope, key]--;
        }

        public void IncrementAll(CounterKey key, T? exclude = null)
        {
            using (MultipleChangeScope())
            {
                foreach (var scope in _scopes)
                {
                    if (exclude.HasValue && exclude.Value.Equals(scope))
                    {
                        continue;
                    }
                    Increment(scope, key);
                }
            }
        }

        public void DecrementAll(CounterKey key)
        {
            using (MultipleChangeScope())
            {
                foreach (var scope in _scopes)
                {
                    Decrement(scope, key);
                }
            }
        }

        bool ICounter.IsSupportScope(CounterScope scope)
        {
            return _supportedBaseScopes[(int)scope];
        }

        void ICounter.ResetAll(CounterScope scope)
        {
            if (_supportedBaseScopes[(int)scope])
            {
                ResetAll(ToCurrent(scope));
            }
        }

        void ICounter.ResetAll(Func<CounterScope, CounterKey, bool> predicate)
        {
            using (MultipleChangeScope())
            {
                foreach (var scope in _scopes)
                {
                    var baseScope = ToBase(scope);
                    var dict = _scopesCounters[ToNumber(scope)];
                    if (dict.Count == 0) continue;
                    var keys = ListPool<string>.Get();
                    keys.AddRange(dict.Keys);
                    foreach (var key in keys.Where(key => predicate(baseScope, CounterKey.FromString(key))))
                    {
                        dict.Remove(key);
                    }
                    ListPool<string>.Release(keys);
                }
            }
        }

        public void Reset(T scope, CounterKey key)
        {
            _scopesCounters[ToNumber(scope)].Remove(key.ToString());
        }

        public void ResetAll(T scope)
        {
            _scopesCounters[ToNumber(scope)].Clear();
        }

        private unsafe ushort ToNumber(T enumValue)
        {
            return *(ushort*)(&enumValue);
        }

        private unsafe CounterScope ToBase(T enumValue)
        {
            return *(CounterScope*)(&enumValue);
        }

        private unsafe T ToCurrent(CounterScope enumValue)
        {
            return *(T*)(&enumValue);
        }
    }
}