using System;
using System.Collections.Generic;
using System.Linq;
using Magify.Model;
using JetBrains.Annotations;

namespace Magify
{
    internal class ContentProvider : IContextListener
    {
        private const string TagEarliest = "earliest";
        private const string TagLatest = "latest";

        [CanBeNull]
        private Dictionary<string, ContentGroup> _content;
        [NotNull]
        private readonly object _lock = new();

        [CanBeNull]
        internal Dictionary<string, ContentGroup> Content => _content;

        public bool HasContent
        {
            get
            {
                lock (_lock)
                {
                    return _content != null;
                }
            }
        }

        public ConfigScope SuitableScope => ConfigScope.Content;

        public void UpdateContext([NotNull] CampaignsContext context, ContextKind kind)
        {
            switch (kind)
            {
                case ContextKind.Default when !HasContent:
                case ContextKind.Saved or ContextKind.Downloaded:
                    lock (_lock)
                    {
                        _content = context.Content;
                    }
                    break;
            }
        }

        public void ResetContentMap()
        {
            lock (_lock)
            {
                _content = default;
            }
        }

        [NotNull]
        public IList<ContentItem> GetContentList([NotNull] string groupKey, [NotNull] string contentKey, [CanBeNull] ICollection<string> tags = null)
        {
            lock (_lock)
            {
                return GetContentListThreadUnsafe(groupKey, contentKey, tags);
            }
        }

        [CanBeNull]
        public ContentItem GetEarliestContent(string groupKey, string contentKey)
        {
            return GetContentItem(groupKey, contentKey, TagEarliest);
        }

        [CanBeNull]
        public ContentItem GetLatestContent(string groupKey, string contentKey)
        {
            return GetContentItem(groupKey, contentKey, TagLatest);
        }

        [NotNull]
        private IList<ContentItem> GetContentListThreadUnsafe(string groupKey, string contentKey, ICollection<string> tags = null)
        {
            if (_content == null || !_content.TryGetValue(groupKey, out var group) || group == null || group.Items == null)
            {
                return Array.Empty<ContentItem>();
            }
            return group.Items.Where(itemFilter).Select(ModelToContentItem).ToList();

            bool itemFilter(Model.ContentItem it)
            {
                if (it.Key != contentKey || it.Tags == null)
                {
                    return false;
                }

                return tags == null || tags.All(it.Tags.Contains);
            }
        }

        [CanBeNull]
        private ContentItem GetContentItem(string groupKey, string contentKey, string tag)
        {
            lock (_lock)
            {
                if (_content == null || !_content.TryGetValue(groupKey, out var group) || group == null)
                {
                    return null;
                }
                var contentList = GetContentListThreadUnsafe(groupKey, contentKey);
                return contentList.Count switch
                {
                    1 => contentList.First(),
                    _ => contentList.FirstOrDefault(c => c.Tags.Contains(tag)),
                };
            }
        }

        private static ContentItem ModelToContentItem([NotNull] Model.ContentItem model) => new()
        {
            Key = model.Key,
            Link = model.Link,
            Previews = model.Previews,
            Tags = model.Tags,
        };
    }
}