using System;
using System.Collections.Generic;
using System.Linq;
using Magify.Model;
using JetBrains.Annotations;
using Newtonsoft.Json;
using UnityEngine;
using UnityEngine.Pool;
using Event = Magify.Model.Event;

namespace Magify
{
    internal class CampaignsProvider
    {
        private record SuitableCampaign(ICampaign Campaign, SourceType SourceType, [CanBeNull] IReadOnlyList<NestedCampaign> FilteredNestedCampaigns, [CanBeNull] AbstractProductsCollection ParsedProducts);

        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        private readonly CampaignsTracker _campaignTracker;
        private readonly CampaignsCollection _campaigns;
        private readonly GeneralPrefs _prefs;
        private readonly Counters _counters;
        private readonly LimitsHolder _limitsHolder;
        private readonly PlatformAPI _platform;
        private readonly LtoCampaignManager _ltoCampaignManager;

        public CampaignsProvider(GeneralPrefs prefs, CampaignsTracker campaignTracker, CampaignsCollection campaigns, Counters counters, LimitsHolder limitsHolder, PlatformAPI platformAPI, LtoCampaignManager ltoCampaignManager)
        {
            _prefs = prefs;
            _campaignTracker = campaignTracker;
            _campaigns = campaigns;
            _counters = counters;
            _limitsHolder = limitsHolder;
            _platform = platformAPI;
            _ltoCampaignManager = ltoCampaignManager;
        }

        [CanBeNull]
        public CampaignRequest ProvideCampaign(string eventName, [CanBeNull] IReadOnlyDictionary<string, object> eventParams)
        {
            _logger.Log($"Search campaign for event: {eventName}");
            _campaignTracker.TrackEvent(eventName);

            var result = FindSuitableCampaign(eventName, eventParams, isDefault: false, withLtoLaunch: true);
            var isDefaultCampaign = result == null;

            if (result == null)
            {
                _logger.Log("Search in DEFAULT context");
                result = FindSuitableCampaign(eventName, eventParams, isDefault: true, withLtoLaunch: true);
            }

            if (result == null)
            {
                var record = _ltoCampaignManager.FindActiveOfferBySpot(eventName);
                if (record != null)
                {
                    _logger.Log("Suitable: TRUE - active offer on spot.");
                    result = CreateCampaignFromModel(record, SourceType.Placement);
                }
            }

            if (result == null)
            {
                return null;
            }

            var customParams = DictionaryPool<string, object>.Get();
            customParams!.Copy(eventParams);

            var source = new CampaignSource(eventName, result.SourceType);
            var campaign = result.Campaign;
            var request = new CampaignRequest
            {
                Campaign = campaign,
                NestedCampaigns = result.FilteredNestedCampaigns,
                ParsedProducts = result.ParsedProducts,
                Source = source,
                CustomParams = customParams,
                IsDefaultCampaign = isDefaultCampaign,
            };

            _campaignTracker.RegisterCampaignRequest(request);
            return request;
        }

        [CanBeNull]
        public ICampaign GetCampaignSilently(string eventName, IReadOnlyDictionary<string, object> eventParams)
        {
            _logger.Log($"{nameof(GetCampaignSilently)} for event={eventName}");

            _campaignTracker.TrackEvent(eventName);
            var result = FindSuitableCampaign(eventName, eventParams, isDefault: false, withLtoLaunch: false);
            result ??= FindSuitableCampaign(eventName, eventParams, isDefault: true, withLtoLaunch: false);
            _campaignTracker.RevertEvent(eventName);

            return result?.Campaign;
        }

        public bool IsCampaignAvailable(string campaignName, string eventName, Dictionary<string, object> eventParams)
        {
            _logger.Log($"{nameof(IsCampaignAvailable)} for event={eventName}");
            var campaign = GetCampaignSilently(eventName, eventParams);
            return campaign?.Name == campaignName;
        }

        [CanBeNull]
        private SuitableCampaign FindSuitableCampaign(string eventName, [CanBeNull] IReadOnlyDictionary<string, object> eventParams, bool isDefault, bool withLtoLaunch)
        {
            _logger.Log($"Provide campaign for {eventName}. IsDefaultContext={isDefault};\nParams:\n{JsonFacade.SerializeObject(eventParams, Formatting.Indented)}");

            var campaignsRecords = isDefault
                ? _campaigns.DefaultCampaigns
                : _campaigns.CurrentCampaigns;

            if (campaignsRecords.Count == 0)
            {
                _logger.Log("No campaigns in config");
                return null;
            }

            var limits = _limitsHolder.GetActiveLimits();
            foreach (var record in campaignsRecords)
            {
                _logger.Log($"Check campaign: {record.Name}");

                var referrerId = _prefs.ReferrerId.Value;
                if (!record.Info.ReferrerLimits.IsCorresponded(referrerId))
                {
                    _logger.Log($"Suitable: FALSE - restricted by referrer={referrerId}");
                    continue;
                }

                if (record.Campaign.Type!.Value.IsSupportNested() is true && !IterateActiveNestedCampaigns(record.Campaign).Any())
                {
                    _logger.Log("Suitable: FALSE - no active nested products.");
                    continue;
                }

                if (record.Campaign.Type?.IsLimitedOffer() is true && _ltoCampaignManager.HasActiveOfferBySpot(record.Name, eventName))
                {
                    _logger.Log("Suitable: TRUE - on spot.");
                    return CreateCampaignFromModel(record, SourceType.Placement);
                }

                var isSuitableByPlacement = record.Info.Placements?.Contains(eventName) ?? false;
                var isCorrespondsPurchaseLimits = record.Info.PurchaseLimits.IsCorresponded(_prefs.AllUsedProducts);

                if (isSuitableByPlacement)
                {
                    if (record.HasPlacementRestrictedByPurchases() && !isCorrespondsPurchaseLimits)
                    {
                        _logger.Log("Suitable: FALSE - restricted by purchased products.");
                        continue;
                    }

                    _logger.Log("Suitable: TRUE - on placement.");
                    return CreateCampaignFromModel(record, SourceType.Placement);
                }

                if (!isCorrespondsPurchaseLimits)
                {
                    _logger.Log("Suitable: FALSE - restricted by purchased products.");
                    continue;
                }

                var checkPlacementsOnly = isDefault && _campaigns.CurrentCampaigns.Count > 0;
                if (checkPlacementsOnly)
                {
                    _logger.Log("Suitable: FALSE - checked only by placement.");
                    continue;
                }

                if (!record.Info.IsWithinClickLimits(_counters.Clicks))
                {
                    _logger.Log("Suitable: FALSE - campaign click limits reached.");
                    continue;
                }

                if (!record.Info.SubscriptionStatus.IsCorresponded(_prefs.SubscriptionStatus.Value))
                {
                    _logger.Log("Suitable: FALSE - on subscription status.");
                    continue;
                }

                if (!record.Info.InAppStatus.IsCorresponded(_prefs.InAppStatus.Value))
                {
                    _logger.Log("Suitable: FALSE - on inApp status.");
                    continue;
                }

                if (!record.Info.PayingStatus.IsCorresponded(_prefs.SubscriptionStatus.Value, _prefs.InAppStatus.Value))
                {
                    _logger.Log("Suitable: FALSE - on paying status.");
                    continue;
                }

                if (!record.Info.AuthorizationStatus.IsCorresponded(_prefs.AuthorizationStatus.Value))
                {
                    _logger.Log("Suitable: FALSE - on authorization status.");
                    continue;
                }

                if (!record.Campaign.IsCampaignRelevant(_platform))
                {
                    _logger.Log("Suitable: FALSE - is not relevant.");
                    continue;
                }

                if (!record.Info.IsWithinRewardsGrantLimits(_counters.Rewards))
                {
                    _logger.Log("Suitable: FALSE - reward watched limits reached.");
                    continue;
                }

                if (!record.Info.IsWithinBonusGrantLimits(_counters.Bonuses))
                {
                    _logger.Log("Suitable: FALSE - free bonus limits reached.");
                    continue;
                }

                var isWithinCampaignImpressionLimits = record.Info.IsWithinCampaignImpressionLimits(_counters.Impressions, _campaignTracker.ImpressionsTime);
                var isWithinGlobalImpressionLimits = MeetsGlobalLimits(limits) && MeetsCampaignTypeLimits(limits, record.Type);

                if (record.Campaign.Type?.IsLimitedOffer() is true && !_ltoCampaignManager.HasActiveOffer(record.Campaign.Name))
                {
                    if (!withLtoLaunch)
                    {
                        _logger.Log("Suitable: FALSE - non active LTO.");
                        continue;
                    }

                    var trigger = record.Campaign.Triggers?.FirstOrDefault(t => t.Id == eventName);
                    if (trigger == null || !IsSuitableByTrigger(record.Campaign, trigger, eventParams))
                    {
                        _logger.Log("Suitable: FALSE - on lto trigger.");
                        continue;
                    }

                    var wasLaunched = _ltoCampaignManager.TryToLaunchOffer(record, trigger);
                    if (!wasLaunched)
                    {
                        _logger.Log("Suitable: FALSE - lto can't be launched.");
                        continue;
                    }
                    _logger.Log("Suitable: TRUE - lto launched.");

                    var launchWithImpression = trigger.ShowOnStart == true;
                    var isWithingTriggerImpressionLimits = MeetsEventImpressionLimits(trigger, record.Name, SourceType.Trigger);
                    var shouldShowCampaign = launchWithImpression && isWithinGlobalImpressionLimits && isWithinCampaignImpressionLimits && isWithingTriggerImpressionLimits;

                    if (!shouldShowCampaign)
                        continue;

                    return CreateCampaignFromModel(record, SourceType.Trigger);
                }

                if (!isWithinGlobalImpressionLimits)
                {
                    _logger.Log("Suitable: FALSE - global impression limits reached.");
                    continue;
                }

                if (!isWithinCampaignImpressionLimits)
                {
                    _logger.Log("Suitable: FALSE - campaign impressions limits reached.");
                    continue;
                }

                if (record.Campaign.Type?.IsLimitedOffer() is true && !record.Campaign.MeetsImpressionsPerActivationLimit(_counters.Impressions))
                {
                    _logger.Log("Suitable: FALSE - lto campaign impressions per activation limits reached.");
                    continue;
                }

                if (!IsSuitableByEvent(record.Info, eventName, eventParams))
                {
                    _logger.Log("Suitable: FALSE - on event.");
                    continue;
                }

                _logger.Log("Suitable: TRUE - on event.");
                return CreateCampaignFromModel(record, SourceType.Event);
            }

            return null;
        }

        [NotNull]
        private SuitableCampaign CreateCampaignFromModel(CampaignRecord record, SourceType sourceType)
        {
            var filteredNestedCampaigns = GetActiveNestedCampaigns(record.Campaign);
            var parsedCampaign = record.Campaign.CreateCampaignFromModel(filteredNestedCampaigns, new CampaignParser.AdditionalData(_prefs.ClientId.Value));

            var campaign = parsedCampaign.Campaign;
            var products = parsedCampaign.Products;

            return new SuitableCampaign(campaign, sourceType, filteredNestedCampaigns, products);
        }

        [NotNull]
        private IEnumerable<NestedCampaign> IterateActiveNestedCampaigns([NotNull] CampaignModel model)
        {
            return model.NestedCampaigns!.Where(p => MeetsNestedCampaignLimits(p) && IsNestedCampaignRelevant(p) && IsNestedFitsFilters(p));
        }

        [CanBeNull]
        internal List<NestedCampaign> GetActiveNestedCampaigns(CampaignModel model)
        {
            return model.Type!.Value.IsSupportNested() is true ? IterateActiveNestedCampaigns(model).ToList() : null;
        }

        internal bool IsNestedCampaignRelevant(NestedCampaign nestedCampaign)
        {
            return nestedCampaign.Type switch
            {
                ProductType.CrossPromo => !_platform.IsApplicationInstalled(nestedCampaign.PromotedApplication),
                _ => true
            };
        }

        private bool IsNestedFitsFilters(NestedCampaign nestedCampaign)
        {
            var subscriptionStatus = _prefs.SubscriptionStatus.Value;
            var inAppStatus = _prefs.InAppStatus.Value;
            var authorizationStatus = _prefs.AuthorizationStatus.Value;
            var referrerId = _prefs.ReferrerId.Value;
            var allUsedProducts = _prefs.AllUsedProducts;

            return nestedCampaign.SubscriptionStatus?.IsCorresponded(subscriptionStatus) is not false
                && nestedCampaign.InAppStatus?.IsCorresponded(inAppStatus) is not false
                && nestedCampaign.AuthorizationStatus?.IsCorresponded(authorizationStatus) is not false
                && nestedCampaign.PayingStatus?.IsCorresponded(subscriptionStatus, inAppStatus) is not false
                && nestedCampaign.GetReferrerLimits().IsCorresponded(referrerId)
                && nestedCampaign.GetPurchaseLimits().IsCorresponded(allUsedProducts);
        }

        private bool MeetsNestedCampaignLimits(NestedCampaign nestedCampaign)
        {
            var keyNested = CounterKey.GetKey(nestedName: nestedCampaign.Name);

            var global = isWithinLimit(NestedCounterScope.Global);
            var session = isWithinLimit(NestedCounterScope.Session);
            var daily = isWithinLimit(NestedCounterScope.Daily);
            var activation = isWithinLimit(NestedCounterScope.Activation);

            return global && session && daily && activation && isWithinPurchaseLimits();

            bool isWithinLimit(NestedCounterScope scope)
            {
                var limitValue = nestedCampaign.GetLimitForScope(scope);
                if (limitValue == null)
                {
                    return true;
                }

                return _counters.Nested[scope, keyNested] < limitValue;
            }

            bool isWithinPurchaseLimits()
            {
                if (nestedCampaign.Type == ProductType.NonConsumable)
                {
                    return !_prefs.AllUsedProducts.Contains(nestedCampaign.ProductId);
                }
                return true;
            }
        }

        private bool MeetsGlobalLimits(LimitsModel limits)
        {
            var keySourceType = CounterKey.GetKey(sourceType: SourceType.Event);

            var globalImpressionTime = _campaignTracker.GetLastImpressionTimestampByLimitedSource(null);
            var isGlobalIntervalExpired = IsTimeIntervalPassed(limits.GlobalInterval, globalImpressionTime);
            var meetsImpressionPerSessionLimit = limits.ImpressionsPerSession == null || _counters.Impressions[ImpressionsCounterScope.Session, keySourceType] < limits.ImpressionsPerSession;

            return isGlobalIntervalExpired && meetsImpressionPerSessionLimit;
        }

        private static bool IsTimeIntervalPassed(TimeSpan? timeInterval, DateTime? impressionTime)
        {
            if (timeInterval == null || impressionTime == null)
            {
                return true;
            }

            return DateTime.UtcNow - impressionTime > timeInterval;
        }

        private bool MeetsCampaignTypeLimits(LimitsModel limits, CampaignType type)
        {
            var campaignImpressionTime = _campaignTracker.GetLastImpressionTimestampByLimitedSource(type);
            var isSessionIntervalExpired = IsTimeIntervalPassed(limits.GetSessionImpressionsInterval(type), campaignImpressionTime);

            var impressionsLimit = limits.CampaignTypeLimits.GetValueOrDefault(type);
            var meetsCampaignTypeImpressions = MeetsCampaignTypeImpressionLimits(type, impressionsLimit);

            var isRelativeIntervalExpired = IsTimeIntervalPassed(limits.GetRelativeImpressionsInterval(type), _campaignTracker.GetRelativeImpressionTimestamp(type));

            return isSessionIntervalExpired && isRelativeIntervalExpired && meetsCampaignTypeImpressions;
        }

        private bool MeetsCampaignTypeImpressionLimits(CampaignType type, [CanBeNull] ImpressionLimitsModel impressionLimits)
        {
            if (impressionLimits == null)
            {
                return true;
            }

            var keySourceTypeTrigger = CounterKey.GetKeyCampaignTypeBySourceType(type, SourceType.Trigger);
            var keySourceTypeEvent = CounterKey.GetKeyCampaignTypeBySourceType(type, SourceType.Event);

            var global = isWithinLimit(impressionLimits.GlobalLimit, ImpressionsCounterScope.Global);
            var session = isWithinLimit(impressionLimits.SessionLimit, ImpressionsCounterScope.Session);
            var daily = isWithinLimit(impressionLimits.DailyLimit, ImpressionsCounterScope.Daily);
            var period = MeetsCampaignTypeLimitsPerPeriod(type, impressionLimits);
            var tempPeriod = MeetsCampaignTypeTemporaryLimits(type, impressionLimits.TempLimits);

            return global && session && daily && period && tempPeriod;

            bool isWithinLimit(int? limitValue, ImpressionsCounterScope scope)
            {
                if (limitValue == null)
                {
                    return true;
                }

                var triggerImpressions = _counters.Impressions[scope, keySourceTypeTrigger];
                var eventImpressions = _counters.Impressions[scope, keySourceTypeEvent];
                return triggerImpressions + eventImpressions < limitValue;
            }
        }

        private bool MeetsCampaignTypeLimitsPerPeriod(CampaignType type, ImpressionLimitsModel limits)
        {
            if (!limits.PeriodLimit.HasValue || !limits.Period.HasValue)
            {
                return true;
            }

            var keySourceTypeTrigger = CounterKey.GetKeyCampaignTypeBySourceType(type, SourceType.Trigger);
            var keySourceTypeEvent = CounterKey.GetKeyCampaignTypeBySourceType(type, SourceType.Event);

            var triggerImpressions = ImpressionUtils.GetImpressionsPerPeriod
            (
                _campaignTracker.ImpressionsTime,
                keySourceTypeTrigger.ToString(),
                limits.Period.Value
            );

            var eventImpressions = ImpressionUtils.GetImpressionsPerPeriod
            (
                _campaignTracker.ImpressionsTime,
                keySourceTypeEvent.ToString(),
                limits.Period.Value
            );

            return triggerImpressions + eventImpressions < limits.PeriodLimit;
        }

        private bool MeetsCampaignTypeTemporaryLimits(CampaignType type, List<TemporaryImpressionLimit> limits)
        {
            return limits.TrueForAll(limit =>
            {
                var impressions = _counters.Impressions[ImpressionsCounterScope.Period, CounterKey.GetKeyCampaignTypeByPeriodLimit(type, limit.LimitId)];
                return impressions < limit.ImpressionsPerPeriod;
            });
        }

        private bool IsSuitableByEvent(CampaignInfo campaign, string eventName, IReadOnlyDictionary<string, object> eventParams)
        {
            var targetEvent = campaign.Events?.FirstOrDefault(e => e.Id == eventName);
            return targetEvent != null &&
                   IsSuitableBySource(campaign.Name, SourceType.Event, targetEvent, eventParams) &&
                   MeetsEventImpressionLimits(targetEvent, campaign.Name, SourceType.Event);
        }

        private bool IsSuitableByTrigger(CampaignModel campaign, Trigger trigger, IReadOnlyDictionary<string, object> eventParams)
        {
            return IsSuitableBySource(campaign.Name, SourceType.Trigger, trigger, eventParams) && MeetsTriggerActivationLimits(trigger, campaign.Name);
        }

        private bool IsSuitableBySource(string campaignName, SourceType type, Event source, IReadOnlyDictionary<string, object> sourceParams)
        {
            return source.IsSuitable(_counters.Events, sourceParams, _prefs.GlobalSessionCounter.Value) &&
                   MeetsEventRewardGrantedLimits(source, campaignName, type) &&
                   MeetsEventBonusGrantedLimits(source, campaignName, type) &&
                   MeetsEventCampaignClickLimits(source, campaignName, type);
        }

        private bool MeetsEventImpressionLimits(Event sourceEvent, string campaignName, SourceType sourceType)
        {
            var counterKey = CounterKey.GetKeyCampaignBySource(campaignName, sourceEvent.Id, sourceType);

            var global = isWithinLimit(sourceEvent.GlobalLimit, ImpressionsCounterScope.Global);
            var session = isWithinLimit(sourceEvent.SessionLimit, ImpressionsCounterScope.Session);
            var daily = isWithinLimit(sourceEvent.DailyLimit, ImpressionsCounterScope.Daily);

            var period = true;
            if (sourceEvent.PeriodLimit.HasValue && sourceEvent.Period.HasValue)
            {
                period = ImpressionUtils.GetImpressionsPerPeriod(_campaignTracker.ImpressionsTime, counterKey.ToString(), sourceEvent.Period.Value) < sourceEvent.PeriodLimit;
            }

            return global && session && daily && period;

            bool isWithinLimit(int? limitValue, ImpressionsCounterScope scope)
            {
                if (limitValue == null)
                    return true;

                return _counters.Impressions[scope, counterKey] < limitValue;
            }
        }

        private bool MeetsTriggerActivationLimits(Trigger trigger, string campaignName)
        {
            var counterKey = CounterKey.GetKeyCampaignBySource(campaignName, trigger.Id);

            var global = isWithinLimit(trigger.ActivationGlobalLimit, LtoStartCounterScope.Global);
            var session = isWithinLimit(trigger.ActivationSessionLimit, LtoStartCounterScope.Session);
            var daily = isWithinLimit(trigger.ActivationDailyLimit, LtoStartCounterScope.Daily);

            var period = true;
            if (trigger.ActivationPeriodLimit != null && trigger.ActivationPeriod != null)
            {
                period = _ltoCampaignManager.GetActivationsPerPeriod(counterKey, trigger.ActivationPeriod.Value) < trigger.ActivationPeriodLimit;
            }

            return global && session && daily && period;

            bool isWithinLimit(int? limitValue, LtoStartCounterScope scope)
            {
                if (limitValue == null)
                    return true;

                return _counters.LtoStart[scope, counterKey] < limitValue;
            }
        }

        private bool MeetsEventRewardGrantedLimits(Event sourceEvent, string campaignName, SourceType sourceType)
        {
            var counterKey = CounterKey.GetKeyCampaignBySource(campaignName, sourceEvent.Id, sourceType);

            var global = isWithinLimit(sourceEvent.GlobalRewardLimit, RewardsCounterScope.Global);
            var session = isWithinLimit(sourceEvent.SessionRewardLimit, RewardsCounterScope.Session);
            var daily = isWithinLimit(sourceEvent.DailyRewardLimit, RewardsCounterScope.Daily);

            return global && session && daily;

            bool isWithinLimit(int? limitValue, RewardsCounterScope scope)
            {
                if (limitValue == null)
                    return true;

                return _counters.Rewards[scope, counterKey] < limitValue;
            }
        }

        private bool MeetsEventBonusGrantedLimits(Event sourceEvent, string campaignName, SourceType sourceType)
        {
            var counterKey = CounterKey.GetKeyCampaignBySource(campaignName, sourceEvent.Id, sourceType);

            var global = isWithinLimit(sourceEvent.GlobalBonusLimit, BonusesCounterScope.Global);
            var session = isWithinLimit(sourceEvent.SessionBonusLimit, BonusesCounterScope.Session);
            var daily = isWithinLimit(sourceEvent.DailyBonusLimit, BonusesCounterScope.Daily);

            return global && session && daily;

            bool isWithinLimit(int? limitValue, BonusesCounterScope scope)
            {
                if (limitValue == null)
                    return true;

                return _counters.Bonuses[scope, counterKey] < limitValue;
            }
        }

        private bool MeetsEventCampaignClickLimits(Event sourceEvent, string campaignName, SourceType sourceType)
        {
            if (sourceEvent.GlobalClickLimit == null)
                return true;

            var counterValue = _counters.Clicks[ClicksCounterScope.Global, CounterKey.GetKeyCampaignBySource(campaignName, sourceEvent.Id, sourceType)];
            return counterValue < sourceEvent.GlobalClickLimit;
        }
    }
}