using System;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class AppStatePrefs : ICancelable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.AppState);

        [NotNull]
        private const string KeyPendingRestore = "pending_restore";
        [NotNull]
        private const string KeyIsAutoRestoreStateEnabled = "is_auto_restore_state_enabled";
        [NotNull]
        private const string KeyCloudClientIdFetched = "client_id_from_cloud_has_been_fetched";
        [NotNull]
        private const string KeyFbAuthorized = "fb_authorized";
        [NotNull]
        private const string KeyExternalAuthProvider = "external_auth_provider";
        [NotNull]
        private const string KeyExternalAuthToken = "external_auth_token";
        [NotNull]
        private const string KeyLastSyncTime = "last_sync_at";

        [NotNull]
        private readonly BinaryStorage _storage;

        public bool IsDisposed { get; private set; }

        [NotNull]
        public IReactiveProperty<bool> IsPendingRestore { get; }
        [NotNull]
        public IReactiveProperty<bool> IsAutoRestoreStateEnabled { get; }
        [NotNull]
        public IReactiveProperty<bool> CloudClientIdFetched { get; }
        [NotNull]
        public IReactiveProperty<bool> IsRestoreInProgress { get; } = new ReactiveProperty<bool>(false);
        [NotNull]
        public IReactiveProperty<bool> IsFbAuthorizationCompleted { get; }
        [NotNull]
        public IReactiveProperty<string> ExternalAuthProvider { get; }
        [NotNull]
        public IReactiveProperty<string> ExternalAuthToken { get; }
        [NotNull]
        public IReactiveProperty<long> LastSyncTime { get; }

        private AppStatePrefs([NotNull] BinaryStorage storage)
        {
            _storage = storage;
            using (MultipleChangeScope())
            {
                IsPendingRestore = _storage.GetReactiveProperty<bool>(KeyPendingRestore);
                IsAutoRestoreStateEnabled = _storage.GetReactiveProperty<bool>(KeyIsAutoRestoreStateEnabled, true);
                CloudClientIdFetched = _storage.GetReactiveProperty<bool>(KeyCloudClientIdFetched);
                IsFbAuthorizationCompleted = _storage.GetReactiveProperty<bool>(KeyFbAuthorized);
                ExternalAuthProvider = _storage.GetReactiveProperty<string>(KeyExternalAuthProvider);
                ExternalAuthToken = _storage.GetReactiveProperty<string>(KeyExternalAuthToken);
                LastSyncTime = _storage.GetReactiveProperty<long>(KeyLastSyncTime, 0);
            }
        }

        [NotNull]
        public static AppStatePrefs Create([NotNull] string storagePath)
        {
            var storage = BinaryStorage
                .Construct(storagePath)
                .AddPrimitiveTypes()
                .SupportNullable<long>()
                .Build();

            var prefs = new AppStatePrefs(storage);
            return prefs;
        }

        void IDisposable.Dispose()
        {
            if (IsDisposed)
            {
                return;
            }
            IsDisposed = true;
            _storage.Dispose();
        }

        public void UpdateExternalAuthToken([CanBeNull] string provider, [CanBeNull] string token)
        {
            ExternalAuthProvider.Value = provider;
            ExternalAuthToken.Value = token;
            _logger.Log($"External auth token updated: {nameof(provider)}: {provider}; {nameof(token)}: {token}");
        }

        public void ResetExternalAuthToken()
        {
            ExternalAuthProvider.Value = null;
            ExternalAuthToken.Value = null;
            _logger.Log("External auth token reset");
        }

        [CanBeNull]
        public (string Provider, string Token)? GetExternalAuthToken()
        {
            if (string.IsNullOrEmpty(ExternalAuthProvider.Value) || string.IsNullOrEmpty(ExternalAuthToken.Value))
                return null;
            return (ExternalAuthProvider.Value, ExternalAuthToken.Value);
        }

        /// <summary>
        /// If you are planning to change many properties of this class, you should use this method.
        /// This method temporarily blocks the writing prefs file to disk inside <b>using</b> construction.
        /// </summary>
        /// <example>
        /// <code>
        /// using (MultipleChangeScope())
        /// {
        ///     AppStatePrefs.IsRestoreInProgress.Value = false;
        ///     AppStatePrefs.IsFbAuthorizationCompleted.Value = true;
        /// }
        /// </code>
        /// </example>
        /// <returns>IDisposable which must disposed automatically after <b>using</b> construction (preferred) or disposed manually.</returns>
        [NotNull]
        public IDisposable MultipleChangeScope()
        {
            return _storage.MultipleChangeScope();
        }

        public void Reset()
        {
            _storage.ResetAll();
        }
    }
}