﻿using JetBrains.Annotations;
using Magify.Rx;

namespace Magify.Features
{
    internal class FeaturesUpdater : IInitializable, ICancelable
    {
        [NotNull]
        private readonly FeaturesProvider _featuresProvider;
        [NotNull]
        private readonly StoredAppFeaturesProvider _storedAppFeaturesProvider;
        [NotNull]
        private readonly GeneralPrefs _generalPrefs;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();

        public bool IsDisposed => _disposables.IsDisposed;

        public FeaturesUpdater(
            [NotNull] FeaturesProvider featuresProvider,
            [NotNull] StoredAppFeaturesProvider storedAppFeaturesProvider,
            [NotNull] GeneralPrefs generalPrefs)
        {
            _featuresProvider = featuresProvider;
            _storedAppFeaturesProvider = storedAppFeaturesProvider;
            _generalPrefs = generalPrefs;
        }

        public void Dispose()
        {
            if (IsDisposed)
            {
                return;
            }
            _disposables.Release();
        }

        void IInitializable.Initialize()
        {
            if (IsDisposed)
            {
                throw new MagifyObjectDisposedException(nameof(FeaturesUpdater));
            }
            _generalPrefs.SubscriptionStatus
                .SkipLatestValueOnSubscribe()
                .Subscribe(UpdateFeaturesOnTrigger)
                .AddTo(_disposables);

            _generalPrefs.InAppStatus
                .SkipLatestValueOnSubscribe()
                .Subscribe(UpdateFeaturesOnTrigger)
                .AddTo(_disposables);

            _generalPrefs.AuthorizationStatus
                .SkipLatestValueOnSubscribe()
                .Subscribe(UpdateFeaturesOnTrigger)
                .AddTo(_disposables);

            _generalPrefs.ReferrerId
                .SkipLatestValueOnSubscribe()
                .Subscribe(UpdateFeaturesOnTrigger)
                .AddTo(_disposables);
        }

        public void OnProductPurchase()
        {
            UpdateFeaturesOnTrigger(0);
        }

        private void UpdateFeaturesOnTrigger<T>([CanBeNull] T _)
        {
            _featuresProvider.UpdateFeaturesOnTrigger();
            _storedAppFeaturesProvider.UpdateFeaturesOnTrigger();
        }
    }
}