using System;
using System.Collections.Generic;
using System.Threading;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify.Aghanim
{
    internal class AghanimPrefs : ICancelable
    {
        [NotNull]
        private const string KeyOrdersMap = "orders_map";
        [NotNull]
        private const string KeyOrderCurrencyMap = "order_currency_map";
        [NotNull]
        private const string KeyOrderPriceMap = "order_price_map";
        [NotNull]
        private const string KeyOrderToProductMap = "order_to_product_map";
        [NotNull]
        private const string KeyOrderToUrlMap = "order_to_url_map";

        [NotNull]
        private readonly BinaryStorage _storage;
        [NotNull]
        private readonly IDictionary<string, AghanimOrderStatus> _ordersMap;
        [NotNull]
        private readonly IDictionary<string, string> _orderCurrencyMap;
        [NotNull]
        private readonly IDictionary<string, string> _orderPriceMap;
        private int _disposeFlag;

        public bool IsDisposed => _disposeFlag > 0;

        // ReSharper disable once InconsistentlySynchronizedField
        // because we never remove keys from this map, we only read or set values
        [NotNull]
        public IEnumerable<string> Orders => _ordersMap.Keys;

        [NotNull]
        public IDictionary<string, string> OrderToProductMap { get; private set; }
        [NotNull]
        public IDictionary<string, string> OrderToUrlMap { get; private set; }

        private AghanimPrefs([NotNull] BinaryStorage storage)
        {
            _storage = storage;
            using (_storage.MultipleChangeScope())
            {
                _ordersMap = _storage.GetDictionary<string, AghanimOrderStatus>(KeyOrdersMap);
                _orderCurrencyMap = _storage.GetDictionary<string, string>(KeyOrderCurrencyMap);
                _orderPriceMap = _storage.GetDictionary<string, string>(KeyOrderPriceMap);
                OrderToProductMap = _storage.GetDictionary<string, string>(KeyOrderToProductMap);
                OrderToUrlMap = _storage.GetDictionary<string, string>(KeyOrderToUrlMap);
            }
        }

        [NotNull]
        public static AghanimPrefs Create([NotNull] string storagePath)
        {
            var storage = BinaryStorage
                .Construct(storagePath)
                .AddPrimitiveTypes()
                .SupportEnum<AghanimOrderStatus>()
                .SupportDictionariesOf<string, AghanimOrderStatus>()
                .SupportDictionariesOf<string, string>()
                .Build();

            var prefs = new AghanimPrefs(storage);
            return prefs;
        }

        public bool TryGetOrderWithStatus([NotNull] string orderId, out AghanimOrderStatus status)
        {
            lock (_ordersMap)
                return _ordersMap.TryGetValue(orderId, out status);
        }

        public AghanimOrderStatus GetOrderStatus([NotNull] string orderId)
        {
            lock (_ordersMap)
                return _ordersMap[orderId];
        }

        public void SetOrderStatus([NotNull] string orderId, AghanimOrderStatus status)
        {
            lock (_ordersMap)
                _ordersMap[orderId] = status;
        }

        public void SaveOrderInfo([NotNull] AghanimOrder order)
        {
            lock (_orderCurrencyMap)
                _orderCurrencyMap[order.Id] = order.Currency;
            lock (_orderPriceMap)
                _orderPriceMap[order.Id] = order.Price;
        }

        public void GetOrderInfo([NotNull] string orderId, [NotNull] out string price, [NotNull] out string currency)
        {
            lock (_orderPriceMap)
                price = _orderPriceMap[orderId];
            lock (_orderCurrencyMap)
                currency = _orderCurrencyMap[orderId];
        }

        [ContractAnnotation("=> true, currency:notnull, price:notnull; => false, currency:null, price:null;")]
        public bool TryGetOrderInfo([NotNull] string orderId, out string currency, out string price)
        {
            lock (_orderCurrencyMap)
            {
                if (!_orderCurrencyMap.TryGetValue(orderId, out currency))
                {
                    price = default;
                    return false;
                }
            }
            lock (_orderPriceMap)
            {
                if (!_orderPriceMap.TryGetValue(orderId, out price))
                {
                    return false;
                }
            }
            return true;
        }

        [NotNull]
        public IDisposable MultipleChangeScope()
        {
            return _storage.MultipleChangeScope();
        }

        void IDisposable.Dispose()
        {
            if (Interlocked.Increment(ref _disposeFlag) == 1)
            {
                _storage.Dispose();
            }
        }
    }
}