using System;
using System.IO;
using System.Threading;
using Cysharp.Threading.Tasks;
using UnityEngine;
using UnityEngine.Networking;
using JetBrains.Annotations;
using Object = UnityEngine.Object;

namespace Magify
{
    public class TextureRemoteStorage : RemoteStorage<Texture>
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Storage);

        public TextureRemoteStorage(string storagePath) : base(storagePath, "textures")
        {
        }

        protected override void ReleaseContent(Texture cache)
        {
            Object.Destroy(cache);
        }

        protected override Texture LoadFromDisk(string filePath)
        {
            var fileData = File.ReadAllBytes(filePath);
            try
            {
                File.SetLastAccessTime(filePath, DateTime.Now);
            }
            catch (Exception e)
            {
                _logger.LogWarning(MagifyLogs.FailedToSetLastAccessTime(filePath, e));
            }
            var texture = GetTextureFromBytes(fileData);
            return texture;
        }

        protected override async UniTask<(StorageResultCode Code, Texture Value)> DownloadFromUrlWithoutTimeout(string url, CancellationToken cancellationToken)
        {
            LogWithPath($"{nameof(DownloadFromUrlWithoutTimeout)} using {nameof(UnityWebRequestTexture)}", url);
            using var www = UnityWebRequestTexture.GetTexture(url);
            var storageResultCode = await SendWebRequest(www, cancellationToken);
            if (storageResultCode != StorageResultCode.Success)
            {
                return (storageResultCode, null);
            }

            LogWithPath($"Save texture raw data on disk", url);
            var raw = www.downloadHandler.data;
            SaveToDisk(url, raw);

            var texture = GetTextureFromBytes(raw);
            return (StorageResultCode.Success, texture);
        }

        private Texture GetTextureFromBytes(byte[] textureData)
        {
            var texture = new Texture2D(2, 2)
            {
                wrapMode = TextureWrapMode.Clamp // Prevents the appearance of artifacts at the edges of the texture
            };
            texture.LoadImage(textureData);
            return texture;
        }
    }
}