using System;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;
using UnityEngine.Networking;

namespace Magify
{
    public class StoredAppFeaturesRemoteStorage : RemoteStorage<StoredAppFeatureContent>
    {
        private static class Headers
        {
            public const string IfNoneMatch = "If-None-Match";
            public const string ETag = "ETag";
        }
        private static class ResponseCodes
        {
            public const int NotModified = 304;
        }

        [NotNull]
        internal const string RootFolderName = "stored_app_features";
        [NotNull]
        internal const string LoadedFeaturesTagsFileName = "loaded_stored_app_features_tags.json";

        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Storage);

        [NotNull]
        private readonly FileStorage _eTagMapFile;
        [NotNull]
        private readonly Dictionary<string, string> _featureToETagMap;
        [NotNull]
        private readonly string _rootPath;

        public StoredAppFeaturesRemoteStorage(string storagePath) : this(storagePath, new RemoteStorageNetworkClient())
        {
        }

        public StoredAppFeaturesRemoteStorage(string storagePath, [NotNull] IRemoteStorageNetworkClient networkClient)
            : base(storagePath, RootFolderName, networkClient, false)
        {
            _rootPath = Path.Combine(storagePath, RootFolderName);
            _eTagMapFile = new FileStorage(_rootPath);
            var json = _eTagMapFile.Load(LoadedFeaturesTagsFileName) ?? string.Empty;
            _featureToETagMap = JsonFacade.DeserializeObject<Dictionary<string, string>>(json) ?? new Dictionary<string, string>();

            try
            {
                ClearCache();
            }
            catch
            {
                // ignore
            }
        }

        internal override async UniTask<ContentHandle<StoredAppFeatureContent>> Load(string url, double timeout, CancellationToken cancellationToken)
        {
            await TaskScheduler.SwitchToMainThread(cancellationToken);
            LogWithPath("Begin download", url);
            var result = await TryLoadFromRemote(url, timeout, cancellationToken);
            if (result?.Code is not StorageResultCode.Success)
            {
                return TryLoadFromDisk(url) ?? result;
            }
            return result;
        }

        protected override async UniTask<(StorageResultCode Code, StoredAppFeatureContent Value)> DownloadFromUrlWithoutTimeout([NotNull] string url, CancellationToken cancellationToken)
        {
            var path = PathForSavingToDisk(url)!;
            var tempPath = path + ".tmp";
            try
            {
                LogWithPath($"{nameof(DownloadFromUrlWithoutTimeout)} has been called", url);
                using (var www = UnityWebRequest.Get(url)!)
                {
                    www.downloadHandler = new DownloadHandlerFile(tempPath);
                    TryAddETag(www);
                    var storageResultCode = await SendWebRequest(www, cancellationToken);
                    if (storageResultCode != StorageResultCode.Success)
                    {
                        return (storageResultCode, null);
                    }
                    if (www.responseCode == ResponseCodes.NotModified)
                    {
                        LogWithPath($"Resource loading finished with {www.responseCode} (not modified), so we already have actual content", url);
                        return (StorageResultCode.AlreadyDownloaded, null);
                    }
                    var newETag = www.GetResponseHeader(Headers.ETag);
                    if (!string.IsNullOrEmpty(newETag))
                    {
                        SaveETagFor(url, newETag);
                    }
                }

                if (File.Exists(path))
                {
                    File.Delete(path);
                }
                File.Move(tempPath, path);
                LogWithPath($"Creating {nameof(StoredAppFeatureContent)} from downloaded file", url);
                return (StorageResultCode.Success, new StoredAppFeatureContent(path));
            }
            finally
            {
                if (File.Exists(tempPath))
                    File.Delete(tempPath);
            }
        }

        protected override void ReleaseContent(StoredAppFeatureContent cache)
        {
        }

        protected override StoredAppFeatureContent LoadFromDisk(string filePath)
        {
            try
            {
                File.SetLastAccessTime(filePath, DateTime.Now);
            }
            catch (Exception e)
            {
                _logger.LogWarning(MagifyLogs.FailedToSetLastAccessTime(filePath, e));
            }
            return new StoredAppFeatureContent(filePath);
        }

        private void TryAddETag([NotNull] UnityWebRequest request)
        {
            if (request.url != null && _featureToETagMap.TryGetValue(request.url, out var eTag))
            {
                request.SetRequestHeader(Headers.IfNoneMatch, eTag);
                _logger.Log($"ETag added for {request.url}: {eTag}");
                return;
            }
            _logger.Log($"There is no eTag for {request.url}");
        }

        private void SaveETagFor([NotNull] string url, [NotNull] string eTag)
        {
            _featureToETagMap[url] = eTag;
            _eTagMapFile.Save(LoadedFeaturesTagsFileName, JsonFacade.SerializeObject(_featureToETagMap));
        }

        public void Clear()
        {
            _featureToETagMap.Clear();
            _eTagMapFile.Clear(LoadedFeaturesTagsFileName);
            ClearCache(true);
        }

        protected override bool PrepareForCacheFileDeletion([NotNull] FileInfo fileInfo)
        {
            var name = fileInfo.Name;
            foreach (var (featureUrl, etag) in _featureToETagMap)
            {
                if (name == Path.GetFileName(featureUrl))
                {
                    var removed = _featureToETagMap.Remove(featureUrl);
                    if (removed) _eTagMapFile.Save(LoadedFeaturesTagsFileName, JsonFacade.SerializeObject(_featureToETagMap));
                    return removed;
                }
            }
            return false;
        }
    }
}