using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using UnityEngine;

namespace Magify
{
    internal abstract class LocalStorage<TContent>
        where TContent : class
    {
        protected static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Storage);

        private readonly CacheContent<TContent> _cache;
        private readonly string _rootDefaultCreatives;
        private bool _isApplicationQuitting;

        internal LocalStorage()
        {
            Application.quitting += OnApplicationQuit;
            _cache = new CacheContent<TContent>(ReleaseContentInternal);
        }

        internal ContentHandle<TContent> GetFromStreamingAssets(string url)
        {
            _logger.Log($"{nameof(GetFromStreamingAssets)} from {url}");
            var filePath = GetFilePathInStreamingAssets(url);
            if (filePath == default)
            {
                LogWithFileName("Asset don't stored in streaming assets", url);
                return CreateContentHandle(url, StorageResultCode.NoDefaultAsset);
            }

            LogWithFileName("Check is asset stored in cache", filePath);
            if (_cache.Contains(filePath))
            {
                LogWithFileName("Get streaming asset from cache", filePath);
                return CreateContentHandle(filePath, StorageResultCode.Success, _cache.Get(filePath));
            }

            LogWithFileName("Load from streaming assets", filePath);
            var streamingAsset = LoadFromStreamingAssets(filePath);
            if (streamingAsset == null)
            {
                return CreateContentHandle(filePath, StorageResultCode.InvalidAssetError);
            }

            LogWithFileName("Success loading from streaming assets", filePath);
            _cache.Store(filePath, streamingAsset);
            return CreateContentHandle(filePath, StorageResultCode.Success, _cache.Get(filePath));
        }

        internal string GetFilePathInStreamingAssets(string url)
        {
            var fileNameWithoutExtension = EmbeddedCreativesUtilities.GetStreamingAssetNameWithoutExtension(url);
            LogWithFileName("Get file path in streaming assets", fileNameWithoutExtension);
            if (!BetterStreamingAssets.DirectoryExists(PackageInfo.ResourcesFolder))
            {
                return default;
            }

            var files = BetterStreamingAssets.GetFiles(PackageInfo.ResourcesFolder, $"{fileNameWithoutExtension}*", SearchOption.AllDirectories);
            return files.FirstOrDefault();
        }

        private void ReleaseContentInternal(TContent cache)
        {
            if (_isApplicationQuitting) return;
            ReleaseContent(cache);
        }

        protected abstract TContent LoadFromStreamingAssets(string url);
        protected abstract void ReleaseContent(TContent content);

        private ContentHandle<TContent> CreateContentHandle(string url, StorageResultCode code, TContent value = null)
        {
            Action callback = null;
            if (value != null)
            {
                callback = () => _cache.Release(url);
            }
            return new ContentHandle<TContent>(code, value, callback);
        }

        private void OnApplicationQuit()
        {
            _isApplicationQuitting = true;
        }

        [Conditional(MagifyLogger.VerboseLoggingDefine)]
        protected void LogWithFileName(string message, string filePath)
        {
            _logger.Log($"{message}\n{Path.GetFileName(filePath)}");
        }
    }
}