using System;
using System.Collections.Generic;
using JetBrains.Annotations;
using UnityEngine;

namespace Magify
{
    internal class CacheContent<TContent>
        where TContent : class
    {
        private readonly Action<TContent> _releaseCallback;
        [NotNull]
        private readonly Dictionary<string, int> _linksCounter = new();
        [NotNull]
        private readonly Dictionary<string, TContent> _caches = new();
        [NotNull]
        private readonly object _lock = new();
        // ReSharper disable once StaticMemberInGenericType
        [NotNull]
        public static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Content);

        public CacheContent(Action<TContent> releaseCallback = null)
        {
            _releaseCallback = releaseCallback;
        }

        public void Store([NotNull] string url, [CanBeNull] TContent content)
        {
            _logger.Log($"Save content to cache\n{url}");
            lock (_lock)
            {
                _caches.Add(url, content);
                _linksCounter.Add(url, 0);
            }
        }

        public bool TryStore([NotNull] string url, [CanBeNull] TContent content)
        {
            _logger.Log($"Save content to cache\n{url}");
            lock (_lock)
            {
                if (_caches.TryAdd(url, content))
                {
                    _linksCounter.Add(url, 0);
                    return true;
                }
                return false;
            }
        }

        [CanBeNull]
        public TContent Get([NotNull] string url)
        {
            lock (_lock)
            {
                _logger.Log($"Get content from cache\n{url}");
                _linksCounter[url]++;
                _logger.Log($"Get content from cache and increase reference to content became: {_linksCounter[url]}\n{url}");
                return _caches[url];
            }
        }

        [CanBeNull]
        public TContent StoreAndGet([NotNull] string url, [CanBeNull] TContent content)
        {
            _logger.Log($"Save content to cache and get reference\n{url}");
            lock (_lock)
            {
                _caches.Add(url, content);
                _linksCounter.Add(url, 1);
                _logger.Log($"Get saved content from cache and increase reference to content became: {_linksCounter[url]}\n{url}");
                return content;
            }
        }

        public bool Contains([NotNull] string url)
        {
            lock (_lock)
            {
                return _caches.ContainsKey(url);
            }
        }

        public void Release([NotNull] string url)
        {
            lock (_lock)
            {
                var links = --_linksCounter[url];
                _logger.Log($"Decrease amount of reference to content, left {links}\n{url}");
                if (links == 0)
                {
                    RemoveThreadUnsafe(url);
                }
            }
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        private void RemoveThreadUnsafe([NotNull] string url)
        {
            _logger.Log($"Remove content from cache\n{url}");
            var content = _caches[url];
            _caches.Remove(url);
            _linksCounter.Remove(url);
            _releaseCallback?.Invoke(content);
        }
    }
}