using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using TMPro;
using UnityEngine;
using UnityEngine.UI;

namespace Magify
{
    public abstract class LimitedTimeOfferBase : MonoBehaviour
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Lto);

        [SerializeField]
        private GameObject _content;
        [SerializeField]
        private Transform _ltoContainer;
        [SerializeField]
        private string _offerSpot;
        [SerializeField]
        private TextMeshProUGUI _title;
        [SerializeField]
        private Button _button;

        private CancellationTokenSource _loadingLtoToken;
        private LtoView _ltoView;
        private LtoBadgeCreative _badge;
        private bool _needUpdateDefaultView;
        private bool _isDestroyed;
        private bool _isApplicationQuit;

        public LtoInfo Offer { get; private set; }
        protected bool IsDefaultView { get; private set; }

        public string OfferSpot
        {
            get => _offerSpot;
            protected set => _offerSpot = value;
        }

        public TimeSpan OfferTimeLeft => Offer == null
            ? TimeSpan.Zero
            : Offer.EndTime - DateTime.UtcNow;

        private bool LastMinuteTimerIsActive =>
            Offer != null &&
            Offer.BadgeCreative.LastMinutesTimer > 0 &&
            Offer.BadgeCreative.LastMinutesTimer >= OfferTimeLeft.TotalMinutes;

        protected virtual void Awake()
        {
            Initialize().Forget();
        }

        private void OnApplicationQuit()
        {
            _isApplicationQuit = true;
        }

        protected virtual void OnDestroy()
        {
            _isDestroyed = true;
            _button.onClick.RemoveListener(OnButtonClick);
            MagifyManager.Lto.OnAdded -= OnOfferAdd;
            MagifyManager.Lto.OnUpdated -= OnOfferUpdate;
            MagifyManager.Lto.OnRemoved -= OnOfferRemove;
            CleanUp();
            if (!_isApplicationQuit)
            {
                ReleaseLtoView(_ltoView);
            }
        }

        protected virtual void Update()
        {
            if (Offer == null) return;

            if (_needUpdateDefaultView && CanUpdateDefaultView())
            {
                UpdateDefaultView().Forget();
            }

            if (OfferTimeLeft.TotalMilliseconds < 0)
            {
                CleanUp();
                return;
            }

            if (Offer.BadgeCreative.HasTimer || LastMinuteTimerIsActive)
            {
                ShowTimer();
            }
            else
            {
                ShowText();
            }
        }

        protected virtual bool CanUpdateDefaultView()
        {
            return false;
        }

        private async UniTaskVoid UpdateDefaultView()
        {
            _needUpdateDefaultView = false;
            _loadingLtoToken = new CancellationTokenSource();
            var view = await GetLtoView(Offer, _loadingLtoToken.Token);
            if (view == null) return;
            if (IsDefaultView)
            {
                ReleaseLtoView(view);
                return;
            }

            ReleaseLtoView(_ltoView);
            _ltoView = view;
            _ltoView.Show();
        }

        protected abstract void OnButtonClick();

        protected virtual string GetFormattedTitle()
        {
            return Offer.BadgeCreative.LabelText;
        }

        protected virtual string GetFormattedTimer()
        {
            return $"{(int)OfferTimeLeft.TotalHours}:{OfferTimeLeft.Minutes}:{OfferTimeLeft.Seconds}";
        }

        private async UniTaskVoid Initialize()
        {
            _button.onClick.AddListener(OnButtonClick);
            gameObject.SetActive(false);
            _content.SetActive(false);

            while (!MagifyManager.Initialized && !_isDestroyed)
            {
                await UniTask.Yield();
            }

            if (_isDestroyed)
            {
                return;
            }

            MagifyManager.Lto.OnAdded += OnOfferAdd;
            MagifyManager.Lto.OnUpdated += OnOfferUpdate;
            MagifyManager.Lto.OnRemoved += OnOfferRemove;

            foreach (var offer in MagifyManager.Lto.GetActiveLtoOffers())
            {
                OnOfferAdd(offer);
            }
        }

        private void OnOfferAdd(LtoInfo offerItem)
        {
            if (_isDestroyed) return;
            if (offerItem.Spot != _offerSpot) return;

            setup().Forget();

            async UniTaskVoid setup()
            {
                CleanUp();
                await SetupOffer(offerItem);
                if (_ltoView == null)
                {
                    return;
                }
                OfferAdded(offerItem);
            }
        }

        private void OnOfferUpdate(LtoInfo offerItem)
        {
            if (offerItem.Spot != _offerSpot) return;
            update().Forget();

            async UniTaskVoid update()
            {
                if (!offerItem.BadgeCreative.Equals(_badge))
                {
                    CleanUp();
                    await SetupOffer(offerItem);
                }
                if (_ltoView == null) return;
                OfferUpdated(offerItem);
            }
        }

        private void OnOfferRemove(LtoInfo ltoInfo)
        {
            if (ltoInfo.Spot != _offerSpot)
            {
                return;
            }

            CleanUp();
            OfferRemoved(ltoInfo);
        }

        protected virtual void OfferAdded(LtoInfo ltoInfo)
        {
        }

        protected virtual void OfferUpdated(LtoInfo ltoInfo)
        {
        }

        protected virtual void OfferRemoved(LtoInfo ltoInfo)
        {
        }

        protected virtual void CleanUp()
        {
            _loadingLtoToken?.Cancel();
            _loadingLtoToken = null;
            ReleaseLtoView(_ltoView);
            _ltoView = null;
            if (gameObject != null) gameObject.SetActive(false);
            if (_content != null) _content.SetActive(false);
            if (_title != null) _title.text = string.Empty;
            Offer = null;
        }

        private async UniTask SetupOffer(LtoInfo offerItem)
        {
            _loadingLtoToken = new CancellationTokenSource();
            _badge = offerItem.BadgeCreative;

            var view = await GetLtoView(offerItem, _loadingLtoToken.Token);
            if (view == null)
            {
                return;
            }

            _needUpdateDefaultView = IsDefaultView && !CanUpdateDefaultView();
            Offer = offerItem;
            _ltoView = view;
            view.Show();
            _content.SetActive(true);
            gameObject.SetActive(true);
            Update();
        }

        private async UniTask<LtoView> GetLtoView(LtoInfo offerItem, CancellationToken cancellationToken)
        {
            _logger.Log($"Getting LTO view for: object name: {name}, spot: {offerItem.Spot}");
            if (_ltoContainer == null)
            {
                _logger.LogError($"LTO container is not set for: object name: {name}, spot: {offerItem.Spot}");
                return null;
            }
            var view = await MagifyManager.Lto.GetLto(offerItem, _ltoContainer, cancellationToken);
            if (cancellationToken.IsCancellationRequested)
            {
                _logger.Log($"LTO view loading has been cancelled for: object name: {(this != null ? name : "[destroyed]")}, spot: {offerItem.Spot}");
                ReleaseLtoView(view);
                return null;
            }
            if (_ltoContainer == null)
            {
                _logger.LogError($"LTO container has been destroyed during lto view loading for: object name: {(this != null ? name : "[destroyed]")}, spot: {offerItem.Spot}");
                return null;
            }
            if (view != null)
            {
                if (!cancellationToken.IsCancellationRequested)
                {
                    IsDefaultView = false;
                    return view;
                }
                ReleaseLtoView(view);
                return null;
            }

            _logger.Log($"Getting default LTO view for: object name: {name}, spot: {offerItem.Spot}");
            view = MagifyManager.Lto.GetDefaultLto(offerItem, _ltoContainer);
            if (view != null)
            {
                if (!cancellationToken.IsCancellationRequested)
                {
                    IsDefaultView = true;
                    return view;
                }
                ReleaseLtoView(view);
                return null;
            }

            return null;
        }

        private static void ReleaseLtoView(LtoView view)
        {
            if (view == null) return;
            view.ReleaseContent();
            if (view.gameObject != null)
            {
                Destroy(view.gameObject);
            }
        }

        private void ShowText()
        {
            _title.text = GetFormattedTitle();
        }

        private void ShowTimer()
        {
            _title.text = GetFormattedTimer();
        }
    }
}