﻿using System.IO;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;

namespace Magify
{
    public partial class MagifyManager
    {
        public static class Storage
        {
            private static TextureRemoteStorage _remoteTextures;
            private static TextureLocalStorage _localTextures;
            private static BundleRemoteStorage _remoteBundles;
            private static BundleLocalStorage _localBundles;
            private static StoredAppFeaturesRemoteStorage _remoteStoredAppFeatures;
            private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Storage);

            public static bool ImageIsCached(string url)
            {
                return _remoteTextures.IsStoredInCache(url) || _remoteTextures.IsCachedOnDisk(url);
            }

            public static bool BundleIsCached(string url)
            {
                return _remoteBundles.IsStoredInCache(url) || _remoteBundles.IsCachedOnDisk(url);
            }

            public static bool HasDefaultCreative(string url)
            {
                return _localTextures.GetFilePathInStreamingAssets(url) != default;
            }

            public static bool HasDefaultBundle(string url)
            {
                return _localBundles.GetFilePathInStreamingAssets(url) != default;
            }

            public static void Clear()
            {
                // TODO:
                // _image.Clear();
                // _bundle.Clear();
            }

            public static async UniTask<ContentHandle<Texture>> LoadImage(string url, double timeout, CancellationToken cancellationToken = default)
            {
                var contentHandle = await _remoteTextures.Load(url, timeout, cancellationToken);
                cancellationToken.ThrowIfCancellationRequested();
                if (NeedLoadDefaultCreative(contentHandle))
                {
                    _logger.Log($"Can't load creative then try GetFromStreamingAssets\n{Path.GetFileName(url)}");
                    return _localTextures.GetFromStreamingAssets(url);
                }

                return contentHandle;
            }

            public static async UniTask<ContentHandle<AssetBundle>> LoadBundle(string url, double timeout, CancellationToken cancellationToken = default)
            {
                var contentHandle = await _remoteBundles.Load(url, timeout, cancellationToken);
                if (NeedLoadDefaultCreative(contentHandle))
                {
                    return _localBundles.GetFromStreamingAssets(url);
                }

                return contentHandle;
            }

            public static async UniTask<ContentHandle<StoredAppFeatureContent>> LoadStoredAppFeature([NotNull] string url, double timeout, CancellationToken cancellationToken)
            {
                _logger.Log($"{nameof(LoadStoredAppFeature)} called for url: {url} with timeout: {timeout}");
                return await _remoteStoredAppFeatures.Load(url, timeout, cancellationToken);
            }

            /// <summary>
            /// Deletes all downloaded feature content from disk.
            /// </summary>
            /// <remarks>
            /// Be sure to release all content handles before calling this method.
            /// </remarks>
            public static void ClearStoredAppFeatures()
            {
                _remoteStoredAppFeatures.Clear();
            }

            public static ContentHandle<Texture> LoadImageFromStreamingAssets(string url)
            {
                return _localTextures.GetFromStreamingAssets(url);
            }

            internal static void Setup(string storagePath)
            {
                BetterStreamingAssets.Initialize();

                MigrateFromGandalf(storagePath);
                _remoteTextures = new TextureRemoteStorage(storagePath);
                _remoteBundles = new BundleRemoteStorage(_root, storagePath);
                _remoteStoredAppFeatures = new StoredAppFeaturesRemoteStorage(storagePath);
                _localTextures = new TextureLocalStorage();
                _localBundles = new BundleLocalStorage();
            }

            private static void MigrateFromGandalf(string storagePath)
            {
                if (Directory.Exists(storagePath)) return;
                const string oldRootFolder = "com.appcraft.gandalf";
                var oldPath = Path.Combine(Application.persistentDataPath, oldRootFolder);
                if (!Directory.Exists(oldPath)) return;
                Directory.Move(oldPath, storagePath);
            }

            private static bool NeedLoadDefaultCreative<TContent>(ContentHandle<TContent> contentHandle)
                where TContent : class
            {
                return contentHandle.Code != StorageResultCode.Success && contentHandle.Code != StorageResultCode.Canceled;
            }
        }
    }
}