﻿using System.Collections.Generic;
using UnityEngine;

namespace Magify
{
    public partial class MagifyManager
    {
        public static class Logging
        {
            public static bool IsLoggingEnabled
            {
                get
                {
                    ThrowIfMagifyIsNotReady(nameof(IsLoggingEnabled));
                    return MagifyLogger.IsLoggingEnabled;
                }
                set
                {
                    ThrowIfMagifyIsNotReady(nameof(IsLoggingEnabled));
                    if (MagifyLogger.IsLoggingEnabled == value) return;
                    if (value) // means logs were disabled
                    {
                        MagifyLogger.IsLoggingEnabled = true;
                        MagifyPlatformAPI.IsLoggingEnabled = true;
                        _logger.Log($"Logging enabled");
                    }
                    else // means logs were enabled
                    {
                        _logger.Log($"Disable logging");
                        MagifyPlatformAPI.IsLoggingEnabled = false;
                        MagifyLogger.IsLoggingEnabled = false;
                    }
                }
            }

            /// <summary>
            /// All logging scopes are disabled by default
            /// </summary>
            public static void EnableScope(string scope)
            {
                MagifyLogger.EnableTag(scope);
                (_logger ?? MagifyLogger.Get()).Log($"Enabled logging scope: {scope}");
            }

            /// <summary>
            /// All logging scopes are disabled by default
            /// </summary>
            public static void DisableScope(string scope)
            {
                MagifyLogger.DisableTag(scope);
                (_logger ?? MagifyLogger.Get()).Log($"Disabled logging scope: {scope}");
            }

            /// <summary>
            /// All Magify logging scopes will be enabled <br/>
            /// Uses <see cref="LoggingScope.IterateScopes"/>>
            /// </summary>
            public static void EnableAllScopes()
            {
                foreach (var scope in LoggingScope.IterateScopes())
                {
                    EnableScope(scope);
                    (_logger ?? MagifyLogger.Get()).Log($"Enabled logging scope: {scope}");
                }
            }

            public static bool IsScopeActive(string scope)
            {
                return MagifyLogger.IsTagActive(scope);
            }

            public static IEnumerable<string> GetActiveScopes()
            {
                return MagifyLogger.GetActiveTags();
            }

            public static void SetCustomPrefixSource(GetCustomLogPrefixDelegate source)
            {
                MagifyLogger.CustomPrefixSource = source;
            }

            public static void Reset()
            {
                MagifyLogger.Reset();
            }
        }
    }
}