using System;
using System.Collections.Generic;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    public partial class MagifyManager
    {
        public static class Features
        {
            /// <remarks>
            /// Since config processing and features parsing are multithreaded operations, this callback is called on the same thread as the processing.
            /// You should switch to main thread manually.
            /// </remarks>
            public static event Action<ConfigKind> OnFeaturesParsed;
            /// <remarks>
            /// Since config processing and features parsing are multithreaded operations, this callback is called on the same thread as the processing.
            /// You should switch to main thread manually.
            /// </remarks>
            public static event Action OnFeaturesUpdated;
            private static IDisposable _onUpdateSubscription;

            public static bool IsCurrentFeaturesParsed
            {
                get
                {
                    ThrowIfMagifyIsNotReady(nameof(IsCurrentFeaturesParsed));
                    return MagifyPlatformAPI.Features.IsCurrentFeaturesParsed;
                }
            }

            /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
            [NotNull]
            public static IStoredAppFeaturesCollection StoredAppFeatures
            {
                get
                {
                    ThrowIfMagifyIsNotReady(nameof(StoredAppFeatures));
                    return MagifyPlatformAPI.StoredAppFeatures;
                }
            }

            internal static void Initialize([NotNull] MagifyPlatformAPI magifyPlatformAPI)
            {
                ReSubscribeOnPlatformAPI(magifyPlatformAPI);
                OnMagifyPlatformAPIChanged += ReSubscribeOnPlatformAPI;
            }

            private static void ReSubscribeOnPlatformAPI([CanBeNull] MagifyPlatformAPI newMagifyPlatformAPI)
            {
                _onUpdateSubscription?.Dispose();
                if (newMagifyPlatformAPI != null)
                {
                	_onUpdateSubscription = Observable
                    	.FromEvent<ConfigKind>(h => newMagifyPlatformAPI.Features.OnUpdate += h, h => newMagifyPlatformAPI.Features.OnUpdate -= h)
                    	.Subscribe(configKind =>
                		{
                    		OnFeaturesParsed?.Invoke(configKind);
                        	OnFeaturesUpdated?.Invoke();
                    	});
                }
            }

            /// <summary>
            /// For correct events work it require to call Initialize() again (with actual MagifyPlatformAPI)
            /// </summary>
            internal static void Dispose()
            {
                OnFeaturesUpdated = null;
                OnMagifyPlatformAPIChanged -= ReSubscribeOnPlatformAPI;
            }

            public static bool TryGetBool(string featureName, out Feature<bool> result)
            {
                ThrowIfMagifyIsNotReady(nameof(TryGetBool));
                return MagifyPlatformAPI.Features.TryGetBool(featureName, out result);
            }

            public static Feature<bool> GetBool(string featureName)
            {
                ThrowIfMagifyIsNotReady(nameof(GetBool));
                return MagifyPlatformAPI.Features.GetBool(featureName);
            }

            public static bool TryGetNumber(string featureName, out Feature<double> result)
            {
                ThrowIfMagifyIsNotReady(nameof(TryGetNumber));
                return MagifyPlatformAPI.Features.TryGetNumber(featureName, out result);
            }

            public static Feature<double> GetNumber(string featureName)
            {
                ThrowIfMagifyIsNotReady(nameof(GetNumber));
                return MagifyPlatformAPI.Features.GetNumber(featureName);
            }

            public static bool TryGetString(string featureName, out Feature<string> result)
            {
                ThrowIfMagifyIsNotReady(nameof(TryGetString));
                return MagifyPlatformAPI.Features.TryGetString(featureName, out result);
            }

            public static Feature<string> GetString(string featureName)
            {
                ThrowIfMagifyIsNotReady(nameof(GetString));
                return MagifyPlatformAPI.Features.GetString(featureName);
            }

            /// <summary>
            /// Allows you to ignore features from a specific source (config)
            /// </summary>
            /// <param name="features">Collection of features which will be ignoring. If null or empty - then all features from a specific source (config) will be available</param>
            /// <param name="source">Type of source (config) from which features will be ignored</param>
            public static void SetIgnoredFeatures([CanBeNull] IEnumerable<string> features, FeatureSource source)
            {
                ThrowIfMagifyIsNotReady(nameof(SetIgnoredFeatures));
                MagifyPlatformAPI.Features.SetIgnoredFeatures(features, source);
            }
        }
    }
}