using System;
using System.Collections.Generic;
using JetBrains.Annotations;

namespace Magify
{
    internal class MagifyWorkersManager : IInitializable, IDisposable
    {
        [NotNull]
        private readonly IReadOnlyCollection<IMagifyWorker> _workers;

        public MagifyWorkersManager([NotNull] IReadOnlyCollection<IMagifyWorker> workers)
        {
            _workers = workers;
        }

        void IInitializable.Initialize()
        {
            foreach (var worker in _workers)
            {
                (worker as IInitializable)?.Initialize();
            }
        }

        void IDisposable.Dispose()
        {
            foreach (var worker in _workers)
            {
                (worker as IDisposable)?.Dispose();
            }
        }

        public void DoJob<T>()
            where T : IMagifyWorkerJob
        {
            DoJob(default(T));
        }

        public void DoJob<T>([NotNull] T job)
            where T : IMagifyWorkerJob
        {
            foreach (var worker in _workers)
            {
                if (worker is IMagifyWorker<T> target)
                {
                    target.DoJob(job);
                    if (target.BreakExecution)
                    {
                        break;
                    }
                }
            }
        }

        public bool DoJob<T, TR>([NotNull] T job, [CanBeNull] out TR result)
            where T : IMagifyWorkerJob
        {
            foreach (var worker in _workers)
            {
                if (worker is IMagifyWorker<T, TR> target)
                {
                    result = target.DoJob(job);
                    return true;
                }
            }
            result = default;
            return false;
        }
    }
}