using System;
using System.Runtime.CompilerServices;
using JetBrains.Annotations;
using Newtonsoft.Json;
using UnityEngine.Scripting;

namespace Magify
{
    /// <summary>
    /// This structure is used to describe a verified purchase that the Magify SDK can trust and not request verification on the Magify services side. <br/>
    /// Magify expects this model to be created after the purchase has been verified and all data from the verification service is already present, which can be reliably displayed in monetization reports. <br/>
    /// This structure has two principal uses: <br/>
    /// <ul>
    ///     <li>
    ///         For the track of the initial purchase. That is, when the user purchased a product. This can be a subscription or in-app. But if one subscription was renewed, it doesn't count for the initial purchase. On the other hand, if a user repeatedly buys consumable products - then all individual purchases will count for the initial purchase.
    ///     </li>
    ///     <li>
    ///         To send additional purchasing information. For example, information about renewing or canceling a subscription. Or to send refund information for an in-app purchase.
    ///     </li>
    /// </ul>
    /// It is highly recommended not to use constructors of this class, but instead use static constructor methods that cover all possible and necessary for Magify events related to Trusted purchases.
    /// </summary>
    [Preserve]
    public class TrustedPurchaseRecord
    {
        /// <summary>
        /// This enum allows to specify the environment for trusted purchase.
        /// </summary>
        [Preserve]
        public enum TrustedPurchaseEnvironment
        {
            Production,
            Sandbox,
        }

        [JsonProperty("product_id"), NotNull, Preserve]
        public string ProductId { get; internal set; }

        [JsonProperty("transaction_id"), NotNull, Preserve]
        public string TransactionId { get; internal set; }

        [JsonProperty("original_transaction_id"), NotNull, Preserve]
        public string OriginalTransactionId { get; init; }

        /// <summary>
        /// The time at which the purchase was made. Format: number seconds by UTC from the start of the Unix epoch, e.g.: <code>DateTimeOffset.UtcNow.ToUnixTimeMilliseconds() / 1000.0d</code>
        /// </summary>
        [JsonProperty("purchased_at"), Preserve]
        public double PurchasedAt { get; init; }

        [JsonProperty("price"), NotNull, Preserve]
        public string Price { get; init; }

        [JsonProperty("currency"), NotNull, Preserve]
        public string Currency { get; init; }

        /// <summary>
        /// Determines trusted purchase environment. This field can be null, then the default value will be Production
        /// </summary>
        [JsonProperty("environment"), CanBeNull, Preserve]
        public TrustedPurchaseEnvironment? Environment { get; set; }

        /// <summary>
        /// ISO 3166-1 alpha-2 code of the region for which the store where the purchase was made is configured.
        /// </summary>
        /// <remarks>
        /// This value will not always match the user's geographical location.
        /// </remarks>
        /// <example>
        /// A user has country “AE” set in their App Store settings, but physically they may be in country “BY”, in this case this field should contain “AE”.
        /// </example>
        [JsonProperty("storefront"), CanBeNull, Preserve]
        public string StoreFront { get; init; }

        /// <summary>
        /// The name of the store where the purchase was made. It can take any value, but it is highly recommended to use one of our suggested values from <see cref="PurchaseStore"/>.
        /// If your store is not in the list - please give us your feedback, and we will add it as soon as possible.
        /// Of course, you can use <see cref="PurchaseStore.Custom"/>, but using predefined names can prevent confusion in reports.
        /// </summary>
        [JsonProperty("store_name"), Preserve]
        public PurchaseStore StoreName { get; set; }

        [JsonProperty("type"), Preserve]
        public PurchaseType Type { get; init; }

        [JsonProperty("period_type"), Preserve]
        public PeriodType PeriodType { get; init; }

        /// <summary>
        /// This field is needed to indicate that the current purchase event describes the transition of a trial subscription to a paid subscription.
        /// You only need to use it once, at the moment when the previous subscription was in the trial period, and all subsequent ones will already be paid.
        /// For paid subscriptions renewing, this field must Not contain `true`.
        /// </summary>
        [JsonProperty("is_trial_conversion"), CanBeNull, Preserve]
        public bool? IsTrialConversion { get; init; }

        /// <summary>
        /// Clarifies exactly what type of purchase was made.
        /// </summary>
        [JsonProperty("product_id_type"), Preserve]
        public ProductIdType ProductIdType { get; init; }

        // ReSharper disable once NotNullOrRequiredMemberIsNotInitialized
        // Because it's for deserialization only
        [Preserve]
        public TrustedPurchaseRecord()
        {
        }

        // ReSharper disable once InvalidXmlDocComment
        // ReSharper disable once NotNullOrRequiredMemberIsNotInitialized
        /// <param name="purchasedAt">The time at which the purchase was made. Format: time offset by UTC, e.g.: <see cref="DateTimeOffset.UtcNow"/></param>
        /// <param name="storeFront">ISO 3166-1 alpha-2 code of the region for which the store where the purchase was made is configured. This value will not always match the user's geographical location.</param>
        [Preserve]
        private TrustedPurchaseRecord(DateTimeOffset purchasedAt)
        {
        }

        // ReSharper disable once InvalidXmlDocComment
        // ReSharper disable once NotNullOrRequiredMemberIsNotInitialized
        /// <param name="purchasedAt"> The time at which the purchase was made. Format: number seconds by UTC from the start of the Unix epoch, e.g.: <code>DateTimeOffset.UtcNow.ToUnixTimeMilliseconds() / 1000.0d</code></param>
        /// <param name="storeFront">ISO 3166-1 alpha-2 code of the region for which the store where the purchase was made is configured. This value will not always match the user's geographical location.</param>
        [Preserve]
        private TrustedPurchaseRecord(int purchasedAt)
        {
        }

        /// <summary>
        /// Constructor for creating a TrustedPurchaseRecord instance with only required fields.<br/>
        /// It is highly recommended not to use constructors of this class, but instead use static constructor methods that cover all possible and necessary for Magify events related to Trusted purchases.
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(DateTimeOffset)" select="param"/>
        [Preserve]
        public TrustedPurchaseRecord(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            DateTimeOffset purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store,
            PurchaseType type) : this(productId, transactionId, originalTransactionId, purchasedAt.ToTimestamp(), price, currency, storeFront, store, type)
        {
        }

        /// <summary>
        /// Constructor for creating a TrustedPurchaseRecord instance with only required fields.<br/>
        /// It is highly recommended not to use constructors of this class, but instead use static constructor methods that cover all possible and necessary for Magify events related to Trusted purchases.
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(int)" select="param"/>
        [Preserve]
        public TrustedPurchaseRecord(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            double purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store,
            PurchaseType type)
        {
            ProductId = productId;
            TransactionId = transactionId;
            OriginalTransactionId = originalTransactionId;
            PurchasedAt = purchasedAt;
            Price = price;
            Currency = currency;
            StoreFront = storeFront;
            StoreName = store;
            Type = type;
        }

        /// <summary>
        /// Constructor for creating a TrustedPurchaseRecord instance with only required fields.<br/>
        /// It is highly recommended not to use constructors of this class, but instead use static constructor methods that cover all possible and necessary for Magify events related to Trusted purchases.
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(DateTimeOffset)" select="param"/>
        [Preserve]
        public TrustedPurchaseRecord(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            DateTimeOffset purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store,
            PurchaseType type,
            PeriodType periodType,
            bool isTrialConversion,
            ProductIdType productIdType) : this(productId, transactionId, originalTransactionId, purchasedAt.ToTimestamp(), price, currency, storeFront, store, type, periodType, isTrialConversion, productIdType)
        {
        }

        /// <summary>
        /// Constructor for creating a TrustedPurchaseRecord instance with only required fields.<br/>
        /// It is highly recommended not to use constructors of this class, but instead use static constructor methods that cover all possible and necessary for Magify events related to Trusted purchases.
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(int)" select="param"/>
        [Preserve]
        public TrustedPurchaseRecord(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            double purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store,
            PurchaseType type,
            PeriodType periodType,
            bool isTrialConversion,
            ProductIdType productIdType)
        {
            ProductId = productId;
            TransactionId = transactionId;
            OriginalTransactionId = originalTransactionId;
            PurchasedAt = purchasedAt;
            Price = price;
            Currency = currency;
            StoreFront = storeFront;
            StoreName = store;
            Type = type;
            PeriodType = periodType;
            IsTrialConversion = isTrialConversion;
            ProductIdType = productIdType;
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>consumable inApp</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(DateTimeOffset)" select="param"/>
        public static TrustedPurchaseRecord CreateForConsumableInApp(
            [NotNull] string productId,
            [NotNull] string transactionId,
            DateTimeOffset purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            return CreateForConsumableInApp(productId, transactionId, purchasedAt.ToTimestamp(), price, currency, storeFront, store);
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>consumable inApp</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(int)" select="param"/>
        public static TrustedPurchaseRecord CreateForConsumableInApp(
            [NotNull] string productId,
            [NotNull] string transactionId,
            double purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            ValidateParams(storeFront);
            return new TrustedPurchaseRecord()
            {
                ProductId = productId,
                TransactionId = transactionId,
                OriginalTransactionId = transactionId,
                PurchasedAt = purchasedAt,
                Price = price,
                Currency = currency,
                StoreFront = storeFront,
                StoreName = store,
                Type = PurchaseType.NoneRenewingPurchase,
                PeriodType = PeriodType.Normal,
                IsTrialConversion = false,
                ProductIdType = ProductIdType.ConsumableProductId,
            };
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>non-consumable inApp</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(DateTimeOffset)" select="param"/>
        public static TrustedPurchaseRecord CreateForNonConsumableInApp(
            [NotNull] string productId,
            [NotNull] string transactionId,
            DateTimeOffset purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            return CreateForNonConsumableInApp(productId, transactionId, purchasedAt.ToTimestamp(), price, currency, storeFront, store);
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>non-consumable inApp</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(int)" select="param"/>
        public static TrustedPurchaseRecord CreateForNonConsumableInApp(
            [NotNull] string productId,
            [NotNull] string transactionId,
            double purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            ValidateParams(storeFront);
            return new TrustedPurchaseRecord()
            {
                ProductId = productId,
                TransactionId = transactionId,
                OriginalTransactionId = transactionId,
                PurchasedAt = purchasedAt,
                Price = price,
                Currency = currency,
                StoreFront = storeFront,
                StoreName = store,
                Type = PurchaseType.NoneRenewingPurchase,
                PeriodType = PeriodType.Normal,
                IsTrialConversion = false,
                ProductIdType = ProductIdType.NonConsumableProductId,
            };
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>initial subscription</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(DateTimeOffset)" select="param"/>
        public static TrustedPurchaseRecord CreateForInitialPaidSubscription(
            [NotNull] string productId,
            [NotNull] string transactionId,
            DateTimeOffset purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            return CreateForInitialPaidSubscription(productId, transactionId, purchasedAt.ToTimestamp(), price, currency, storeFront, store);
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>initial subscription</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(int)" select="param"/>
        public static TrustedPurchaseRecord CreateForInitialPaidSubscription(
            [NotNull] string productId,
            [NotNull] string transactionId,
            double purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            ValidateParams(storeFront);
            return new TrustedPurchaseRecord()
            {
                ProductId = productId,
                TransactionId = transactionId,
                OriginalTransactionId = transactionId,
                PurchasedAt = purchasedAt,
                Price = price,
                Currency = currency,
                StoreFront = storeFront,
                StoreName = store,
                Type = PurchaseType.InitialPurchase,
                PeriodType = PeriodType.Normal,
                IsTrialConversion = false,
                ProductIdType = ProductIdType.SubscriptionProductId,
            };
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>initial trial subscription</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(DateTimeOffset)" select="param"/>
        public static TrustedPurchaseRecord CreateForInitialTrialSubscription(
            [NotNull] string productId,
            [NotNull] string transactionId,
            DateTimeOffset purchasedAt,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            return CreateForInitialTrialSubscription(productId, transactionId, purchasedAt.ToTimestamp(), storeFront, store);
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>initial trial subscription</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(int)" select="param"/>
        public static TrustedPurchaseRecord CreateForInitialTrialSubscription(
            [NotNull] string productId,
            [NotNull] string transactionId,
            double purchasedAt,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            ValidateParams(storeFront);
            return new TrustedPurchaseRecord()
            {
                ProductId = productId,
                TransactionId = transactionId,
                OriginalTransactionId = transactionId,
                PurchasedAt = purchasedAt,
                Price = string.Intern("0"),
                Currency = string.Empty,
                StoreFront = storeFront,
                StoreName = store,
                Type = PurchaseType.InitialPurchase,
                PeriodType = PeriodType.Trial,
                IsTrialConversion = false,
                ProductIdType = ProductIdType.SubscriptionProductId,
            };
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>first paid subscription after trial</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(DateTimeOffset)" select="param"/>
        public static TrustedPurchaseRecord CreateForTrialToPaidSubscriptionTransition(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            DateTimeOffset purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            return CreateForTrialToPaidSubscriptionTransition(productId, transactionId, originalTransactionId, purchasedAt.ToTimestamp(), price, currency, storeFront, store);
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>first paid subscription after trial</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(int)" select="param"/>
        public static TrustedPurchaseRecord CreateForTrialToPaidSubscriptionTransition(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            double purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            ValidateParams(storeFront);
            return new TrustedPurchaseRecord()
            {
                ProductId = productId,
                TransactionId = transactionId,
                OriginalTransactionId = originalTransactionId,
                PurchasedAt = purchasedAt,
                Price = price,
                Currency = currency,
                StoreFront = storeFront,
                StoreName = store,
                Type = PurchaseType.Renewal,
                PeriodType = PeriodType.Normal,
                IsTrialConversion = true,
                ProductIdType = ProductIdType.SubscriptionProductId,
            };
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>renewal subscription</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(DateTimeOffset)" select="param"/>
        public static TrustedPurchaseRecord CreateForSubscriptionRenewal(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            DateTimeOffset purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            return CreateForSubscriptionRenewal(productId, transactionId, originalTransactionId, purchasedAt.ToTimestamp(), price, currency, storeFront, store);
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>renewal subscription</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(int)" select="param"/>
        public static TrustedPurchaseRecord CreateForSubscriptionRenewal(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            double purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            ValidateParams(storeFront);
            return new TrustedPurchaseRecord()
            {
                ProductId = productId,
                TransactionId = transactionId,
                OriginalTransactionId = originalTransactionId,
                PurchasedAt = purchasedAt,
                Price = price,
                Currency = currency,
                StoreFront = storeFront,
                StoreName = store,
                Type = PurchaseType.Renewal,
                PeriodType = PeriodType.Normal,
                IsTrialConversion = false,
                ProductIdType = ProductIdType.SubscriptionProductId,
            };
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>subscription refund</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(DateTimeOffset)" select="param"/>
        public static TrustedPurchaseRecord CreateForSubscriptionRefund(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            DateTimeOffset purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            return CreateForSubscriptionRefund(productId, transactionId, originalTransactionId, purchasedAt.ToTimestamp(), price, currency, storeFront, store);
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>subscription refund</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(int)" select="param"/>
        public static TrustedPurchaseRecord CreateForSubscriptionRefund(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            double purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            ValidateParams(storeFront);
            return new TrustedPurchaseRecord()
            {
                ProductId = productId,
                TransactionId = transactionId,
                OriginalTransactionId = originalTransactionId,
                PurchasedAt = purchasedAt,
                Price = price,
                Currency = currency,
                StoreFront = storeFront,
                StoreName = store,
                Type = PurchaseType.Cancellation,
                PeriodType = PeriodType.Normal,
                IsTrialConversion = false,
                ProductIdType = ProductIdType.SubscriptionProductId,
            };
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>consumable inApp refund</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(DateTimeOffset)" select="param"/>
        public static TrustedPurchaseRecord CreateForConsumableInAppRefund(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            DateTimeOffset purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            return CreateForConsumableInAppRefund(productId, transactionId, originalTransactionId, purchasedAt.ToTimestamp(), price, currency, storeFront, store);
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>consumable inApp refund</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(int)" select="param"/>
        public static TrustedPurchaseRecord CreateForConsumableInAppRefund(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            double purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            ValidateParams(storeFront);
            return new TrustedPurchaseRecord()
            {
                ProductId = productId,
                TransactionId = transactionId,
                OriginalTransactionId = originalTransactionId,
                PurchasedAt = purchasedAt,
                Price = price,
                Currency = currency,
                StoreFront = storeFront,
                StoreName = store,
                Type = PurchaseType.Cancellation,
                PeriodType = PeriodType.Normal,
                IsTrialConversion = false,
                ProductIdType = ProductIdType.ConsumableProductId,
            };
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>non-consumable inApp refund</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(DateTimeOffset)" select="param"/>
        public static TrustedPurchaseRecord CreateForNonConsumableInAppRefund(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            DateTimeOffset purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            return CreateForNonConsumableInAppRefund(productId, transactionId, originalTransactionId, purchasedAt.ToTimestamp(), price, currency, storeFront, store);
        }

        /// <summary>
        /// This method is a template which allows to create TrustedPurchaseRecord with pre-completed fields for <b>non-consumable inApp refund</b>
        /// </summary>
        /// <inheritdoc cref="TrustedPurchaseRecord(int)" select="param"/>
        public static TrustedPurchaseRecord CreateForNonConsumableInAppRefund(
            [NotNull] string productId,
            [NotNull] string transactionId,
            [NotNull] string originalTransactionId,
            double purchasedAt,
            [NotNull] string price,
            [NotNull] string currency,
            [CanBeNull] string storeFront,
            PurchaseStore store)
        {
            ValidateParams(storeFront);
            return new TrustedPurchaseRecord()
            {
                ProductId = productId,
                TransactionId = transactionId,
                OriginalTransactionId = originalTransactionId,
                PurchasedAt = purchasedAt,
                Price = price,
                Currency = currency,
                StoreFront = storeFront,
                StoreName = store,
                Type = PurchaseType.Cancellation,
                PeriodType = PeriodType.Normal,
                IsTrialConversion = false,
                ProductIdType = ProductIdType.NonConsumableProductId,
            };
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        private static void ValidateParams([CanBeNull] string storeFront)
        {
            if (storeFront != null && storeFront.Length != 2)
            {
                throw new MagifyArgumentOutOfRangeException(nameof(storeFront), "null | ISO 3166-1 alpha-2");
            }
        }

        /// <summary>
        /// Allows you to mark the current purchase as if it was made in a test store. This can be useful for testing to clearly separate the sources of purchases.
        /// </summary>
        public void AsSandbox()
        {
            StoreName = PurchaseStore.Sandbox;
        }
    }
}