using System;
using JetBrains.Annotations;
using UnityEngine.Scripting;

namespace Magify
{
    /// <summary>
    /// This structure defines the store where the purchase was made. It can take any value, but it is highly recommended to use one of our suggested values.
    /// If your store is not in the list - please give us your feedback, and we will add it as soon as possible.
    /// Of course, you can use <see cref="PurchaseStore.Custom"/>, but using predefined names can prevent confusion in reports.
    /// </summary>
    /// <remarks>
    /// Be careful when using <see cref="GooglePlay"/> and <see cref="AppStore"/> (or others) for tracking Magify campaigns.
    /// Check <see cref="Native"/>, as this may be more appropriate.
    /// </remarks>
    [Preserve]
    public readonly struct PurchaseStore : IEquatable<PurchaseStore>
    {
        [NotNull, Preserve]
        public string Name { get; }

        /// <summary>
        /// Abstraction over <see cref="GooglePlay"/> and <see cref="AppStore"/>.
        /// If you use Magify's campaigns, then probably you receive this kind of products and should track it.
        /// </summary>
        public static PurchaseStore Native { get; } = new("NATIVE");
        public static PurchaseStore Aghanim { get; } = new("AGHANIM");
        public static PurchaseStore GooglePlay { get; } = new("PLAY_STORE");
        public static PurchaseStore AppStore { get; } = new("APP_STORE");
        public static PurchaseStore Amazon { get; } = new("AMAZON");
        public static PurchaseStore MacAppStore { get; } = new("MAC_APP_STORE");
        public static PurchaseStore MicrosoftStore { get; } = new("MICROSOFT_STORE");
        public static PurchaseStore GalaxyStore { get; } = new("GALAXY_STORE");
        public static PurchaseStore Stripe { get; } = new("STRIPE");
        public static PurchaseStore HuaweiAppGallery { get; } = new("HUAWEI_APPGALLERY");
        public static PurchaseStore MetaQuestStore { get; } = new("META_QUEST_STORE");
        public static PurchaseStore EpicGamesStore { get; } = new("EPIC_GAMES_STORE");
        public static PurchaseStore Steam { get; } = new("STEAM");
        public static PurchaseStore OneStore { get; } = new("ONE_STORE");
        public static PurchaseStore XiaomiGetApps { get; } = new("XIAOMI_GETAPPS");
        public static PurchaseStore TapTap { get; } = new("TAPTAP");
        public static PurchaseStore GetJar { get; } = new("GETJAR");
        public static PurchaseStore MobangoAppStore { get; } = new("MOBANGO");
        public static PurchaseStore SlideMe { get; } = new("SLIDEME");
        public static PurchaseStore Sandbox { get; } = new("SANDBOX");

        [Preserve]
        public PurchaseStore([NotNull] string storeName)
        {
            Name = storeName;
        }

        [Preserve]
        public static PurchaseStore Custom([NotNull] string storeName)
        {
            return new PurchaseStore(storeName);
        }

        public bool Equals(PurchaseStore other)
        {
            return Name == other.Name;
        }

        public override bool Equals(object obj)
        {
            return obj is PurchaseStore other && Equals(other);
        }

        public override int GetHashCode()
        {
            return Name.GetHashCode();
        }

        public static bool operator ==(PurchaseStore left, PurchaseStore right)
        {
            return left.Equals(right);
        }

        public static bool operator !=(PurchaseStore left, PurchaseStore right)
        {
            return !left.Equals(right);
        }
    }
}