using System;
using System.Timers;
using System.Threading;
using JetBrains.Annotations;

namespace Magify.Rx
{
    internal class TimerObservable : OperatorObservableBase<long>
    {
        private readonly TimeSpan _time;
        private readonly bool _useAsInterval;

        public TimerObservable(TimeSpan time, bool useAsInterval)
        {
            _time = time;
            _useAsInterval = useAsInterval;
        }

        protected override IDisposable SubscribeCore(IObserver<long> observer, IDisposable cancel)
        {
            var cts = new CancellationTokenSource();
            var timerObserver = new Timer(cts.Token, _time, _useAsInterval, observer, cancel);

            var disposables = new CompositeDisposable(2);
            disposables.Add(cts);
            disposables.Add(timerObserver);
            return disposables;
        }

        private class Timer : OperatorObserverBase<long, long>
        {
            [NotNull]
            private readonly System.Timers.Timer _timer;
            private readonly CancellationToken _cancellationToken;
            private long _index;

            public Timer(CancellationToken cancellationToken, TimeSpan time, bool useAsInterval, [NotNull] IObserver<long> observer, [NotNull] IDisposable cancel)
                : base(observer, cancel)
            {
                _cancellationToken = cancellationToken;
                _timer = new System.Timers.Timer(time.TotalMilliseconds);
                _timer.AutoReset = useAsInterval;
                _timer.Elapsed += ElapsedHandler;
                _timer.Start();
            }

            private void ElapsedHandler(object sender, ElapsedEventArgs elapsedEventArgs)
            {
                TaskScheduler.RunOnMainThread(OnNext, _cancellationToken).Forget();
            }

            public void OnNext()
            {
                try
                {
                    Observer.OnNext(_index++);
                }
                catch
                {
                    Dispose();
                    throw;
                }
            }

            public override void OnNext(long value)
            {
                // no use.
            }

            public override void OnError(Exception error)
            {
                try { Observer.OnError(error); }
                finally { Dispose(); }
            }

            public override void OnCompleted()
            {
                try { Observer.OnCompleted(); }
                finally { Dispose(); }
            }
        }
    }
}