using System;
using JetBrains.Annotations;

namespace Magify.Rx
{
    internal interface ISelect<out TR>
    {
        [NotNull]
        IObservable<TR> CombinePredicate(Func<TR, bool> predicate);
    }

    internal class SelectObservable<T, TR> : OperatorObservableBase<TR>, ISelect<TR>
    {
        [NotNull]
        private readonly IObservable<T> _source;
        [NotNull]
        private readonly Func<T, TR> _selector;

        public SelectObservable([NotNull] IObservable<T> source, [NotNull] Func<T, TR> selector)
        {
            this._source = source;
            this._selector = selector;
        }

        [NotNull]
        public IObservable<TR> CombinePredicate(Func<TR, bool> predicate)
        {
            return new SelectWhereObservable<T, TR>(this._source, this._selector, predicate);
        }

        [NotNull]
        protected override IDisposable SubscribeCore([NotNull] IObserver<TR> observer, [NotNull] IDisposable cancel)
        {
            return _source.Subscribe(new Select(this, observer, cancel));
        }

        private class Select : OperatorObserverBase<T, TR>
        {
            [NotNull]
            private readonly SelectObservable<T, TR> _parent;

            public Select([NotNull] SelectObservable<T, TR> parent, [NotNull] IObserver<TR> observer, [NotNull] IDisposable cancel) : base(observer, cancel)
            {
                _parent = parent;
            }

            public override void OnNext(T value)
            {
                TR v;
                try
                {
                    v = _parent._selector(value);
                }
                catch (Exception ex)
                {
                    try { Observer.OnError(ex); } finally { Dispose(); }
                    return;
                }

                Observer.OnNext(v);
            }

            public override void OnError(Exception error)
            {
                try { Observer.OnError(error); } finally { Dispose(); }
            }

            public override void OnCompleted()
            {
                try { Observer.OnCompleted(); } finally { Dispose(); }
            }
        }
    }
}