using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;

namespace Magify.Rx
{
    internal class ObserveOnMainThreadObservable<T> : OperatorObservableBase<T>
    {
        [NotNull]
        private readonly IObservable<T> _source;
        private readonly CancellationToken _cancellationToken;

        public ObserveOnMainThreadObservable([NotNull] IObservable<T> source, CancellationToken cancellationToken)
        {
            _source = source;
            _cancellationToken = cancellationToken;
        }

        protected override IDisposable SubscribeCore([NotNull] IObserver<T> observer, [NotNull] IDisposable cancel)
        {
            return _source.Subscribe(new ObserveOnMainThread(observer, cancel, _cancellationToken));
        }

        private class ObserveOnMainThread : OperatorObserverBase<T, T>
        {
            private readonly CancellationToken _cancellationToken;

            public ObserveOnMainThread([NotNull] IObserver<T> observer, [NotNull] IDisposable cancel, CancellationToken cancellationToken)
                : base(observer, cancel)
            {
                _cancellationToken = cancellationToken;
            }

            public override void OnNext(T value)
            {
                if (TaskScheduler.IsMainThread)
                {
                    Observer.OnNext(value);
                    return;
                }

                try
                {
                    TaskScheduler.SwitchToMainThread(_cancellationToken)
                        .ContinueWith(() => Observer.OnNext(value))
                        .Forget(exception =>
                        {
                            try { Observer.OnError(exception!); } finally { Dispose(); }
                        });
                }
                catch (Exception ex)
                {
                    try { Observer.OnError(ex); } finally { Dispose(); }
                }
            }

            public override void OnError(Exception error)
            {
                try { Observer.OnError(error); } finally { Dispose(); }
            }

            public override void OnCompleted()
            {
                try { Observer.OnCompleted(); } finally { Dispose(); }
            }
        }
    }
}