using System;
using System.Threading;
using JetBrains.Annotations;

namespace Magify.Rx
{
    public static partial class Observable
    {
        [NotNull]
        public static IObservable<bool> EveryApplicationFocus()
        {
            return MainThreadDispatcher.OnApplicationFocusAsObservable();
        }

        [NotNull]
        public static IObservable<bool> EveryApplicationPause()
        {
            return MainThreadDispatcher.OnApplicationPauseAsObservable();
        }

        [NotNull]
        public static IDisposable Subscribe<T>([NotNull] this IObservable<T> source, [NotNull] Action<T> onNext)
        {
            return source.Subscribe(Observer.CreateSubscribeObserver(onNext, Stubs.Throw, Stubs.Nop));
        }

        [NotNull]
        public static IDisposable Subscribe<T>([NotNull] this IObservable<T> source, [NotNull] Action<T> onNext, [NotNull] Action<Exception> onError, [NotNull] Action onCompleted)
        {
            return source.Subscribe(Observer.CreateSubscribeObserver(onNext, onError, onCompleted));
        }

        [NotNull]
        public static IObservable<T> ObserveOnMainThread<T>([NotNull] this IObservable<T> source, CancellationToken cancellationToken)
        {
            return new ObserveOnMainThreadObservable<T>(source, cancellationToken);
        }

        [NotNull]
        public static IObservable<Unit> FromEvent(Action<Action> addHandler, Action<Action> removeHandler)
        {
            return new FromEventObservableVoid(addHandler, removeHandler);
        }

        [NotNull]
        public static IObservable<T> FromEvent<T>(Action<Action<T>> addHandler, Action<Action<T>> removeHandler)
        {
            return new FromEventObservable<T>(addHandler, removeHandler);
        }

        [NotNull]
        public static IObservable<T> Skip<T>([NotNull] this IObservable<T> source, int count)
        {
            if (count < 0)
            {
                throw new ArgumentOutOfRangeException(nameof(count));
            }

            // optimize .Skip(count).Skip(count)
            if (source is SkipObservable<T> skip)
            {
                return skip.Combine(count);
            }

            return new SkipObservable<T>(source, count);
        }

        [NotNull]
        public static IObservable<TR> Select<T, TR>([NotNull] this IObservable<T> source, [NotNull] Func<T, TR> selector)
        {
            if (source is WhereObservable<T> whereObservable)
            {
                return whereObservable.CombineSelector(selector);
            }

            return new SelectObservable<T, TR>(source, selector);
        }

        [NotNull]
        public static IObservable<T> Where<T>([NotNull] this IObservable<T> source, [NotNull] Func<T, bool> predicate)
        {
            return source switch
            {
                // optimized path
                WhereObservable<T> whereObservable => whereObservable.CombinePredicate(predicate),
                ISelect<T> selectObservable => selectObservable.CombinePredicate(predicate),
                _ => new WhereObservable<T>(source, predicate),
            };
        }

        [NotNull]
        public static IObservable<long> Interval(TimeSpan interval)
        {
            return new TimerObservable(interval, useAsInterval: true);
        }

        [NotNull]
        public static IObservable<long> Timer(TimeSpan timer)
        {
            return new TimerObservable(timer, useAsInterval: false);
        }

        [NotNull]
        public static IObservable<long> Timer(DateTimeOffset timer)
        {
            return new TimerObservable(timer.UtcDateTime - DateTime.UtcNow, useAsInterval: false);
        }
    }
}